<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller; 
use App\Models\Publicacion;
use App\Models\EstadoPublicacion;
use App\Models\Aprobacion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;


class AprobacionController extends Controller
{
    //------Método para listar publicaciones pendientes--------
    public function listarPendientes(Request $request)
    {
        $estadoPendiente = EstadoPublicacion::where('nombre_estado', 'pendiente')->first();

        if (!$estadoPendiente) {
            return response()->json(['message' => 'Estado pendiente no encontrado'], 404);
        }

        $publicacionesPendientes = Publicacion::where('id_estado_publicacion', $estadoPendiente->id_estado_publicacion)
        ->with('user', 'estado')
        ->orderBy('fecha_creacion', 'desc')
        ->get();

        $publicaciones = $publicacionesPendientes->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'tipo_determinante' => $publicacion->tipo_determinante,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //---------Aprobar una publicación---------
    public function aprobar(Request $request, $id)
    {
        $publicacion = Publicacion::find($id);

        if (!$publicacion) {
            return response()->json(['message' => 'Publicación no encontrada'], 404);
        }

        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();
        $publicacion->id_estado_publicacion = $estadoAprobado->id_estado_publicacion;
        $publicacion->fecha_aprobacion = now();
        $publicacion->save();

        Aprobacion::create([
            'id_publicacion' => $publicacion->id_publicacion,
            'id_user' => auth()->id(),
            'fecha_aprobacion' => now(),
            'estado' => 'aprobado',
            'comentarios' => $request->input('comentarios') ?? null,
        ]);

        return response()->json([
            'message' => 'Publicación aprobada con éxito',
            'contenido' => $publicacion->contenido,
            'url' => $publicacion->url,
        ], 200);
    }

    //------Para subir portada------------
    public function aprobarPortada(Request $request, $id)
    {
        $publicacion = Publicacion::find($id);

        if (!$publicacion) {
            return response()->json(['message' => 'Publicación no encontrada'], 404);
        }

        if ($request->hasFile('portada')) {
            $validatedData = $request->validate([
                'portada' => 'nullable|file|mimes:jpeg,png,jpg|max:20480',
            ]);

            $portadaPath = $request->file('portada')->store('publicaciones_portada', 'public');
            $publicacion->portada = $portadaPath;
            $publicacion->save(); 
        } else {
            return response()->json(['message' => 'No se recibió ninguna imagen de portada.'], 400);
        }

        return response()->json([
            'message' => 'Portada subida con éxito',
            'url_portada' => $this->getPortadaUrl($publicacion),
        ], 200);
    }
    

    //--------Rechazar una publicación (con comentarios)-----------
    public function rechazar(Request $request, $id)
    {
        $publicacion = Publicacion::find($id);
        if (!$publicacion) {
            return response()->json(['message' => 'Publicación no encontrada'], 404);
        }

        $request->validate([
            'comentario' => 'required|string|max:255', 
        ]);

        $estadoRechazado = EstadoPublicacion::where('nombre_estado', 'rechazado')->first();
        $publicacion->id_estado_publicacion = $estadoRechazado->id_estado_publicacion;
        $publicacion->fecha_aprobacion = now(); 
        $publicacion->save();

        Aprobacion::create([
            'id_publicacion' => $publicacion->id_publicacion,
            'id_user' => auth()->id(), 
            'fecha_aprobacion' => now(), 
            'estado' => 'rechazado',
            'comentarios' => $request->comentario, 
        ]);

        return response()->json(['message' => 'Publicación rechazada con éxito'], 200);
    }



    //-------Listar publicaciones rechazadas-----------
    public function listarRechazados(Request $request)
    {
        $estadoRechazado = EstadoPublicacion::where('nombre_estado', 'rechazado')->first();

        if (!$estadoRechazado) {
            return response()->json(['message' => 'Estado rechazado no encontrado'], 404);
        }

        $publicacionesRechazadas = Publicacion::where('id_estado_publicacion', $estadoRechazado->id_estado_publicacion)
            ->with('user', 'estado', 'aprobaciones')
            ->orderBy('fecha_creacion', 'desc')
            ->get();

        $publicaciones = $publicacionesRechazadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            $ultimoComentario = $publicacion->aprobaciones()->where('estado', 'rechazado')->latest()->first();

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'comentario_rechazo' => $ultimoComentario ? $ultimoComentario->comentarios : null, // Agrega el comentario de rechazo
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //---------Listar publicaciones aprobadas-------------
    public function listarAprobados(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
        ->with('user', 'estado')
        ->orderBy('fecha_creacion', 'desc')
        ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //---------Lista la area de Hábitos y estilos de vida saludable-------
    public function listarAprobadosUno(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Hábitos y estilos de vida saludable');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }

 

    //------Lista la area de Acceso seguro a servicios y medicamentos-----
    public function listarAprobadosDos(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Acceso seguro a servicios y medicamentos');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }



    //------Lista la area Salud mental de la población-------
    public function listarAprobadosTres(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Salud mental de la población');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }



    //------Lista la area Salud mental del personal de salud-------
    public function listarAprobadosCuatro(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Salud mental del personal de salud');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area Prevención del embarazo de adolecentes-------
    public function listarAprobadosCinco(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Prevención del embarazo de adolecentes');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area Prevención de la violencia sexual-------
    public function listarAprobadosSeis(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Prevención de la violencia sexual');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area Higiene menstrual-------
    public function listarAprobadosSiete(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Higiene menstrual');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area Vigilancia de la mortalidad materna-------
    public function listarAprobadosOcho(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Vigilancia de la mortalidad materna');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area Control Pre y Post Natal-------
    public function listarAprobadosNueve(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'Control Pre y Post Natal');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }


    //------Lista la area VIH e infecciones de transmisión sexual-------
    public function listarAprobadosDiez(Request $request)
    {
        $estadoAprobado = EstadoPublicacion::where('nombre_estado', 'aprobado')->first();

        if (!$estadoAprobado) {
            return response()->json(['message' => 'Estado aprobado no encontrado'], 404);
        }

        $publicacionesAprobadas = Publicacion::where('id_estado_publicacion', $estadoAprobado->id_estado_publicacion)
            ->whereIn('id_publicacion', function ($query) {
                $query->select('id_publicacion')
                    ->from('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('areas.nombre_area', 'VIH e infecciones de transmisión sexual');
            })
            ->with('user.entidad', 'estado')
            ->orderBy('fecha_creacion', 'desc') 
            ->get();

        $publicaciones = $publicacionesAprobadas->map(function ($publicacion) {
            $estado = $publicacion->estado->nombre_estado; 
            $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;
            
            $urlPortada = $publicacion->portada ? asset("storage/" . $publicacion->portada) : null;

            $areas = DB::table('area_publicacion')
                ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                ->pluck('areas.nombre_area');

            return [
                'id_publicacion' => $publicacion->id_publicacion,
                'tipo_publicacion' => $publicacion->tipo_publicacion,
                'titulo' => $publicacion->titulo,
                'contenido' => $urlContenido,
                'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                'url' => $publicacion->url ?? '',
                'descripcion' => $publicacion->descripcion ?? '',
                'estado_publicacion' => $estado,
                'areas' => $areas,
                'tipo_determinante' => $publicacion->tipo_determinante,
                'entidad_usuario' => $publicacion->user->entidad->nombre_entidad,
                'portada' => $urlPortada,
            ];
        });

        return response()->json($publicaciones, 200);
    }

    protected function getPortadaUrl($publicacion)
    {
        return $publicacion->portada ? asset('storage/' . $publicacion->portada) : null;
    }

    public function AprobadosTotal()
    {
        $total = Aprobacion::where('estado', 'aprobado')->count();
        return response()->json(['total_aprobados' => $total], 200);
    }

    public function RechazadosTotal()
    {
        $total = Aprobacion::where('estado', 'rechazado')->count();
        return response()->json(['total_rechazados' => $total], 200);
    }
}
