<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class CategoriasController extends Controller
{
    /**
     * Listar categorías (incluye sección y proyecto para contexto)
     */
    public function index()
    {
        try {
            $categorias = DB::table('categorias')
                ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
                ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
                ->select(
                    'categorias.id_categoria',
                    'categorias.id_seccion',
                    'secciones.nombre as seccion_nombre',
                    'proyectos.id_proyecto',
                    'proyectos.nombre as proyecto_nombre',
                    'categorias.nombre',
                    'categorias.descripcion',
                    'categorias.orden',
                    'categorias.visible',
                    'categorias.created_at',
                    'categorias.updated_at'
                )
                ->orderBy('categorias.orden', 'asc')
                ->get();

            return response()->json([
                'message' => 'Categorías listadas correctamente.',
                'data' => $categorias
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al listar categorías', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener categorías.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Crear una categoría
     */
    public function store(Request $request)
    {
        Log::info('Inicio de store() en CategoriasController', ['input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'id_seccion' => 'required|integer|exists:secciones,id_seccion',
            'nombre' => 'required|string|max:150',
            'descripcion' => 'nullable|string',
            'orden' => 'nullable|integer',
            'visible' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al crear categoría', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        // Verificar unicidad de nombre dentro de la sección
        $exists = DB::table('categorias')
            ->where('id_seccion', $request->id_seccion)
            ->where('nombre', $request->nombre)
            ->exists();
        if ($exists) {
            return response()->json(['error' => 'Ya existe una categoría con ese nombre en la sección.'], 422);
        }

        DB::beginTransaction();

        $createdDirs = [];

        try {
            $id = DB::table('categorias')->insertGetId([
                'id_seccion' => $request->id_seccion,
                'nombre' => $request->nombre,
                'descripcion' => $request->descripcion ?? null,
                'orden' => $request->orden ?? 0,
                'visible' => $request->filled('visible') ? (bool)$request->visible : true,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            // Obtener id_proyecto de la seccion para construir ruta
            $seccion = DB::table('secciones')->where('id_seccion', $request->id_seccion)->first();
            $id_proyecto_from_seccion = $seccion->id_proyecto ?? null;

            // Si por alguna razón no encontramos el proyecto (muy improbable por el exists), lanzamos excepción
            if (empty($id_proyecto_from_seccion)) {
                throw new \Exception("No se pudo determinar id_proyecto para la sección {$request->id_seccion}");
            }

            // --- Crear carpeta para la categoría dentro de la sección del proyecto ---
            $categoryDir = "proyectos/{$id_proyecto_from_seccion}/secciones/{$request->id_seccion}/categorias/{$id}";
            \Illuminate\Support\Facades\Storage::disk('public')->makeDirectory($categoryDir);
            $createdDirs[] = $categoryDir;
            Log::info("Carpeta creada para categoría: {$categoryDir}");

            DB::commit();
            Log::info('Transacción commit - categoría creada', ['id_categoria' => $id]);

            // --- Obtener la categoría creada con los mismos campos que en index() ---
            $categoria = DB::table('categorias')
                ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
                ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
                ->select(
                    'categorias.id_categoria',
                    'categorias.id_seccion',
                    'secciones.nombre as seccion_nombre',
                    'proyectos.id_proyecto',
                    'proyectos.nombre as proyecto_nombre',
                    'categorias.nombre',
                    'categorias.descripcion',
                    'categorias.orden',
                    'categorias.visible',
                    'categorias.created_at',
                    'categorias.updated_at'
                )
                ->where('categorias.id_categoria', $id)
                ->first();

            // Fallback: si por alguna razón la consulta con join no devuelve, recuperar la fila simple
            if (! $categoria) {
                Log::warning('Consulta con join no devolvió la categoría creada; usando fallback simple.', ['id_categoria' => $id]);
                $categoria = DB::table('categorias')->where('id_categoria', $id)->first();
            }

            Log::info("Categoría creada y recuperada para respuesta", ['id' => $id, 'categoria' => (array)$categoria]);

            return response()->json([
                'message' => 'Categoría creada correctamente.',
                'data' => $categoria
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();

            foreach ($createdDirs as $dir) {
                try {
                    \Illuminate\Support\Facades\Storage::disk('public')->deleteDirectory($dir);
                    Log::info("Carpeta eliminada por rollback: {$dir}");
                } catch (\Exception $ex) {
                    Log::warning("No se pudo eliminar la carpeta en rollback: {$dir}. Error: " . $ex->getMessage());
                }
            }

            Log::error('Error al crear categoría', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al crear categoría.',
                'details' => $e->getMessage()
            ], 500);
        }
    }


    /**
     * Actualizar una categoría
     */
    public function update(Request $request, $id)
    {
        Log::info('Inicio de update() en CategoriasController', ['id' => $id, 'input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'id_seccion' => 'nullable|integer|exists:secciones,id_seccion',
            'nombre' => 'required|string|max:150',
            'descripcion' => 'nullable|string',
            'orden' => 'nullable|integer',
            'visible' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al actualizar categoría', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $row = DB::table('categorias')->where('id_categoria', $id)->first();
            if (! $row) {
                DB::rollBack();
                return response()->json(['error' => "Categoría {$id} no existe."], 404);
            }

            $idSeccion = $request->filled('id_seccion') ? $request->id_seccion : $row->id_seccion;

            // Unicidad nombre dentro de la sección (excluir la propia fila)
            $exists = DB::table('categorias')
                ->where('id_seccion', $idSeccion)
                ->where('nombre', $request->nombre)
                ->where('id_categoria', '!=', $id)
                ->exists();
            if ($exists) {
                DB::rollBack();
                return response()->json(['error' => 'Ya existe una categoría con ese nombre en la sección.'], 422);
            }

            DB::table('categorias')->where('id_categoria', $id)->update([
                'id_seccion' => $idSeccion,
                'nombre' => $request->nombre,
                'descripcion' => $request->descripcion ?? $row->descripcion,
                'orden' => $request->filled('orden') ? $request->orden : $row->orden,
                'visible' => $request->filled('visible') ? (bool)$request->visible : $row->visible,
                'updated_at' => now(),
            ]);

            DB::commit();
            Log::info("Transacción commit - categoría actualizada", ['id_categoria' => $id]);

            // --- Obtener la categoría actualizada con los mismos campos que en index() ---
            $categoria = DB::table('categorias')
                ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
                ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
                ->select(
                    'categorias.id_categoria',
                    'categorias.id_seccion',
                    'secciones.nombre as seccion_nombre',
                    'proyectos.id_proyecto',
                    'proyectos.nombre as proyecto_nombre',
                    'categorias.nombre',
                    'categorias.descripcion',
                    'categorias.orden',
                    'categorias.visible',
                    'categorias.created_at',
                    'categorias.updated_at'
                )
                ->where('categorias.id_categoria', $id)
                ->first();

            // Fallback: si por alguna razón la consulta con join no devuelve, recuperar la fila simple
            if (! $categoria) {
                Log::warning('Consulta con join no devolvió la categoría actualizada; usando fallback simple.', ['id_categoria' => $id]);
                $categoria = DB::table('categorias')->where('id_categoria', $id)->first();
            }

            Log::info("Categoría ID {$id} actualizada (respuesta)", ['before' => $row, 'after' => (array)$categoria]);

            return response()->json([
                'message' => 'Categoría actualizada correctamente.',
                'data' => $categoria
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar categoría', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al actualizar categoría.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
