<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class EntidadController extends Controller
{
    /**
     * Listar todas las entidades.
     */
    public function index()
    {
        try {
            $entidades = DB::table('entidades')
                ->select(
                    'id_entidad',
                    'nombre_entidad',
                    'descripcion',
                    'ciudad',
                    'direccion',
                    'telefono',
                    'created_at',
                    'updated_at'
                )
                ->orderBy('id_entidad', 'desc')
                ->get();

            return response()->json([
                'message' => 'Entidades listadas correctamente.',
                'entidades' => $entidades
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error listando entidades', ['error' => $e->getMessage()]);
            return response()->json([
                'error' => 'Error al listar entidades.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Crear nueva entidad.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'nombre_entidad' => 'required|string|max:100|unique:entidades,nombre_entidad',
            'descripcion'    => 'nullable|string',
            'ciudad'         => 'required|string|max:50',
            'direccion'      => 'required|string|max:150',
            'telefono'       => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al crear entidad', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $id = DB::table('entidades')->insertGetId([
                'nombre_entidad' => $request->nombre_entidad,
                'descripcion'    => $request->descripcion ?? null,
                'ciudad'         => $request->ciudad,
                'direccion'      => $request->direccion,
                'telefono'       => $request->telefono ?? null,
                'created_at'     => $now,
                'updated_at'     => $now,
            ]);

            DB::commit();

            $entidad = DB::table('entidades')->where('id_entidad', $id)->first();

            return response()->json([
                'message' => 'Entidad creada exitosamente.',
                'data' => $entidad
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creando entidad', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al crear entidad.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar entidad.
     */
    public function update(Request $request, $id)
    {
        Log::info('Inicio de update() EntidadController', ['id' => $id, 'input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'nombre_entidad' => "required|string|max:100|unique:entidades,nombre_entidad,{$id},id_entidad",
            'descripcion'    => 'nullable|string',
            'ciudad'         => 'required|string|max:50',
            'direccion'      => 'required|string|max:150',
            'telefono'       => 'nullable|string|max:20',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al actualizar entidad', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $entidad = DB::table('entidades')->where('id_entidad', $id)->first();
            if (!$entidad) {
                Log::error("Entidad no encontrada ID: {$id}");
                return response()->json(['error' => "Entidad {$id} no existe."], 404);
            }

            DB::table('entidades')->where('id_entidad', $id)->update([
                'nombre_entidad' => $request->nombre_entidad,
                'descripcion'    => $request->descripcion ?? null,
                'ciudad'         => $request->ciudad,
                'direccion'      => $request->direccion,
                'telefono'       => $request->telefono ?? null,
                'updated_at'     => Carbon::now(),
            ]);

            DB::commit();

            $updated = DB::table('entidades')->where('id_entidad', $id)->first();

            return response()->json([
                'message' => 'Entidad actualizada correctamente.',
                'data' => $updated
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error actualizando entidad', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al actualizar entidad.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
