<?php

namespace App\Http\Controllers\Api;
use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Http;
use App\Mail\InscripcionVerification; // Mailable que construyes

class InscripcionController extends Controller
{
    public function store(Request $request)
    {
        // 1) Reglas de validación
        $rules = [
            'nombre_completo'       => 'required|string|max:200',
            'email'                 => 'required|email|max:150|unique:inscripciones_360,email',
            'organizacion'          => 'nullable|string|max:200',
            'cargo'                 => 'nullable|string|max:100',
            'pais'                  => 'nullable|string|max:100',
            'ciudad_departamento'   => 'nullable|string|max:100',
            'sector'                => 'nullable|string|max:100',
            'interes_salud_digital' => 'nullable|string',
            'recaptchaToken'        => 'nullable|string', // validamos que llegue el token
        ];

        $messages = [
        'email.unique' => 'Este correo ya ha sido registrado.',
        // Puedes agregar más mensajes personalizados si lo deseas
    ];

        $validator = Validator::make($request->all(), $rules, $messages);

        if ($validator->fails()) {
            Log::warning('Validación de inscripción fallida', [
                'errors' => $validator->errors()->toArray()
            ]);

            return response()->json([
                'success' => false,
                'errors'  => $validator->errors(),
            ], 422);
        }

        // 2) Verificar reCAPTCHA v2-invisible
        $recaptchaSecret   = config('services.recaptcha.secret');
        $recaptchaResponse = $request->input('recaptchaToken');
        $googleRes = Http::asForm()->post(
    'https://www.google.com/recaptcha/api/siteverify',
    [
        'secret'   => config('services.recaptcha.secret'),
        'response' => $recaptchaResponse,
        'remoteip' => $request->ip(),        // opcional, pero recomendado
    ]
)->json();

if (empty($googleRes['success'])) {
    Log::warning('reCAPTCHA falló', ['googleRes' => $googleRes]);
    return response()->json([
        'success' => false,
        'message' => 'Error de validación CAPTCHA.'
    ], 400);
}

        // 3) Preparar datos y token de verificación
        $data      = $validator->validated();
        $token     = Str::random(64);
        $pendingId = DB::table('estados_inscripcion')
                       ->where('clave', 'pending')
                       ->value('id');

        try {
            // 4) Insertar en estado PENDING
            DB::table('inscripciones_360')->insert([
                'estado_inscripcion_id' => $pendingId,
                'nombre_completo'       => $data['nombre_completo'],
                'email'                 => $data['email'],
                'organizacion'          => $data['organizacion'] ?? null,
                'cargo'                 => $data['cargo'] ?? null,
                'pais'                  => $data['pais'] ?? null,
                'ciudad_departamento'   => $data['ciudad_departamento'] ?? null,
                'sector'                => $data['sector'] ?? null,
                'interes_salud_digital' => $data['interes_salud_digital'] ?? null,
                'verification_token'    => $token,
                'token_created_at'      => now(),
                'created_at'            => now(),
                'updated_at'            => now(),
            ]);

            Log::info('Inscripción creada en estado pendiente', [
                'email' => $data['email'],
                'token' => $token,
            ]);

            // 5) Enviar correo de verificación
            Mail::to($data['email'])
                ->send(new InscripcionVerification($token));

            // 6) Responder al front
            return response()->json([
                'success' => true,
                'message' => "Acabamos de enviarte un correo de verificación a {$data['email']}. " .
                             "Por favor revisa tu bandeja de entrada."
            ], 201);

        } catch (\Exception $e) {
            Log::error('Error al crear inscripción', [
                'exception' => $e->getMessage()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Ocurrió un error al procesar la inscripción.'
            ], 500);
        }
    }

    /**
     * Endpoint para confirmar el token de verificación.
     */
    public function confirm(Request $request)
    {
        $token = $request->query('token');

        $record = DB::table('inscripciones_360')
                    ->where('verification_token', $token)
                    ->first();

        if (! $record) {
            return response()->json([
                'success' => false,
                'message' => 'Token inválido.'
            ], 400);
        }

        // 7) Verificar caducidad (24 horas)
        if ($record->token_created_at < now()->subDay()) {
            // Marcar como EXPIRED
            $expiredId = DB::table('estados_inscripcion')
                           ->where('clave', 'expired')
                           ->value('id');

            DB::table('inscripciones_360')
              ->where('id_inscripcion', $record->id_inscripcion)
              ->update(['estado_inscripcion_id' => $expiredId]);

            return response()->json([
                'success' => false,
                'message' => 'El enlace de verificación ha expirado.'
            ], 400);
        }

        // 8) Confirmar y marcar VERIFIED
        $verifiedId = DB::table('estados_inscripcion')
                        ->where('clave', 'verified')
                        ->value('id');

        DB::table('inscripciones_360')
          ->where('id_inscripcion', $record->id_inscripcion)
          ->update([
              'estado_inscripcion_id' => $verifiedId,
              'verification_token'    => null,
              'token_created_at'      => null,
              'updated_at'            => now(),
          ]);

        return response()->json([
            'success' => true,
            'message' => '¡Correo verificado correctamente!'
        ]);
    }
}
