<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProyectosController extends Controller
{
    /**
     * Listar proyectos con sus integrantes (y datos básicos del usuario).
     */
    public function index()
    {
        try {
            // Obtener proyectos (mismos campos que tenías)
            $proyectos = DB::table('proyectos')
                ->select('id_proyecto', 'nombre', 'descripcion', 'created_at', 'updated_at')
                ->orderByDesc('id_proyecto')
                ->get();

            // Mapear proyectos para incluir integrantes con todos los campos relacionados
            // y además secciones + categorias por seccion
            $proyectosConIntegrantes = $proyectos->map(function ($p) {
                // Obtener integrantes y traer todos los campos del integrante y del usuario relacionado,
                // además de información de rol, estado de usuario y entidad (si existen).
                $integrantes = DB::table('integrantes as i')
                    ->leftJoin('users as u', 'i.id_user', '=', 'u.id_user')
                    ->leftJoin('roles as r', 'u.id_rol', '=', 'r.id_rol')
                    ->leftJoin('estado_users as eu', 'u.id_estado_user', '=', 'eu.id_estado_user')
                    ->leftJoin('entidades as en', 'u.id_entidad', '=', 'en.id_entidad')
                    ->where('i.id_proyecto', $p->id_proyecto)
                    ->select(
                        // campos de la tabla integrantes
                        'i.id',
                        'i.id_proyecto',
                        'i.id_user',
                        'i.fecha_ingreso',
                        'i.fecha_salida',
                        'i.created_at as integrante_created_at',
                        'i.updated_at as integrante_updated_at',

                        // campos completos de la tabla users (manteniendo nombres para evitar romper código)
                        'u.id_user as user_id',
                        'u.nombre_usuario',
                        'u.password',
                        'u.ultima_sesion',
                        'u.nombre',
                        'u.apellido_paterno',
                        'u.apellido_materno',
                        'u.telefono',
                        'u.ci',
                        'u.email',
                        'u.id_estado_user',
                        'u.id_entidad',
                        'u.id_rol',
                        'u.remember_token',
                        'u.created_at as user_created_at',
                        'u.updated_at as user_updated_at',

                        // datos adicionales de relaciones
                        'r.id_rol as rol_id',
                        'r.nombre_rol as nombre_rol',

                        'eu.id_estado_user as estado_user_id',
                        'eu.nombre_estado as nombre_estado_user',
                        'eu.descripcion as estado_user_descripcion',

                        'en.id_entidad as entidad_id',
                        'en.nombre_entidad',
                        'en.descripcion as entidad_descripcion',
                        'en.ciudad as entidad_ciudad',
                        'en.direccion as entidad_direccion',
                        'en.telefono as entidad_telefono',
                        'en.created_at as entidad_created_at',
                        'en.updated_at as entidad_updated_at'
                    )
                    ->orderBy('i.fecha_ingreso', 'asc')
                    ->get();

                $p->integrantes = $integrantes;

                // --- Secciones del proyecto ---
                $secciones = DB::table('secciones')
                    ->where('id_proyecto', $p->id_proyecto)
                    ->select(
                        'id_seccion',
                        'id_proyecto',
                        'nombre',
                        'descripcion',
                        'orden',
                        'visible',
                        'created_at',
                        'updated_at'
                    )
                    ->orderBy('orden', 'asc')
                    ->get();

                // Para cada seccion, traer sus categorias
                $seccionesConCategorias = $secciones->map(function ($s) {
                    $categorias = DB::table('categorias')
                        ->where('id_seccion', $s->id_seccion)
                        ->select(
                            'id_categoria',
                            'id_seccion',
                            'nombre',
                            'descripcion',
                            'orden',
                            'visible',
                            'created_at',
                            'updated_at'
                        )
                        ->orderBy('orden', 'asc')
                        ->get();

                    $s->categorias = $categorias;
                    return $s;
                });

                $p->secciones = $seccionesConCategorias;

                return $p;
            });

            return response()->json([
                'message' => 'Proyectos listados correctamente.',
                'data' => $proyectosConIntegrantes
            ], 200);

        } catch (\Exception $e) {
            Log::error('Error al listar proyectos', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al listar proyectos.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
{
    Log::info('Inicio de store() en ProyectosController', ['input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'nombre' => 'required|string|max:150|unique:proyectos,nombre',
        'descripcion' => 'nullable|string',
        'integrantes' => 'nullable|array',
        'integrantes.*.id_user' => 'required_with:integrantes|integer|exists:users,id_user',
        'integrantes.*.fecha_ingreso' => 'nullable|date',
        'integrantes.*.fecha_salida' => 'nullable|date',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al crear proyecto', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors()
        ], 422);
    }

    DB::beginTransaction();
    Log::info('Transacción iniciada para crear proyecto');

    // array para llevar carpetas creadas y poder limpiar en caso de rollback
    $createdDirs = [];

    try {
        // Insertar proyecto
        $idProyecto = DB::table('proyectos')->insertGetId([
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion ?? null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);
        Log::info("Proyecto creado con ID: {$idProyecto}");

        // Insertar integrantes (si vienen)
        $integrantes = $request->integrantes ?? [];
        foreach ($integrantes as $ing) {
            // Evitar duplicados explícitos por id_user dentro del mismo proyecto
            $exists = DB::table('integrantes')
                ->where('id_proyecto', $idProyecto)
                ->where('id_user', $ing['id_user'])
                ->exists();

            if ($exists) {
                Log::warning("Integrante {$ing['id_user']} ya existe en el proyecto {$idProyecto}, se omite inserción");
                continue;
            }

            // Normalizar fechas: aceptar ISO con 'Z', sin Z, yyyy-mm-dd, etc.
            try {
                $fecha_ingreso = isset($ing['fecha_ingreso']) && !empty($ing['fecha_ingreso'])
                    ? Carbon::parse($ing['fecha_ingreso'])->format('Y-m-d H:i:s')
                    : Carbon::now()->format('Y-m-d H:i:s');
            } catch (\Exception $ex) {
                // si parse falla dejamos now()
                $fecha_ingreso = Carbon::now()->format('Y-m-d H:i:s');
            }

            try {
                $fecha_salida = isset($ing['fecha_salida']) && !empty($ing['fecha_salida'])
                    ? Carbon::parse($ing['fecha_salida'])->format('Y-m-d H:i:s')
                    : null;
            } catch (\Exception $ex) {
                $fecha_salida = null;
            }

            DB::table('integrantes')->insert([
                'id_proyecto' => $idProyecto,
                'id_user' => $ing['id_user'],
                'fecha_ingreso' => $fecha_ingreso,
                'fecha_salida' => $fecha_salida,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            Log::info("  • Integrante (user_id: {$ing['id_user']}) agregado al proyecto {$idProyecto}");
        }

        // --- Crear carpeta del proyecto en storage/app/public ---
        $projectDir = "proyectos/{$idProyecto}";
        // makeDirectory es idempotente; si falla lanzará excepción
        \Illuminate\Support\Facades\Storage::disk('public')->makeDirectory($projectDir);
        $createdDirs[] = $projectDir;
        Log::info("Carpeta creada para proyecto: {$projectDir}");

        DB::commit();
        Log::info("Transacción finalizada correctamente. Proyecto ID: {$idProyecto}");

        return response()->json([
            'message' => 'Proyecto creado exitosamente.',
            'id_proyecto' => $idProyecto
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();

        // intentar limpiar carpetas creadas en caso de rollback
        foreach ($createdDirs as $dir) {
            try {
                \Illuminate\Support\Facades\Storage::disk('public')->deleteDirectory($dir);
                Log::info("Carpeta eliminada por rollback: {$dir}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar la carpeta en rollback: {$dir}. Error: " . $ex->getMessage());
            }
        }

        Log::error('Error al crear proyecto, se hizo rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al crear proyecto.',
            'details' => $e->getMessage()
        ], 500);
    }
}

    public function update(Request $request, $id)
{
    Log::info('Inicio de update() en ProyectosController', ['id' => $id, 'input' => $request->all()]);

    $validator = Validator::make(array_merge($request->all(), ['id_proyecto' => $id]), [
        'id_proyecto' => 'required|integer|exists:proyectos,id_proyecto',
        'nombre' => "required|string|max:150|unique:proyectos,nombre,{$id},id_proyecto",
        'descripcion' => 'nullable|string',
        'integrantes' => 'nullable|array',
        'integrantes.*.id' => 'nullable|integer|exists:integrantes,id',
        'integrantes.*.id_user' => 'required_with:integrantes|integer|exists:users,id_user',
        'integrantes.*.fecha_ingreso' => 'nullable|date',
        'integrantes.*.fecha_salida' => 'nullable|date',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al actualizar proyecto', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors()
        ], 422);
    }

    DB::beginTransaction();
    Log::info("Transacción iniciada para update proyecto ID: {$id}");

    try {
        $proyecto = DB::table('proyectos')->where('id_proyecto', $id)->first();
        if (!$proyecto) {
            Log::error("Proyecto no encontrado ID: {$id}");
            return response()->json(['error' => "Proyecto {$id} no existe."], 404);
        }

        // 1) Actualizar cabecera proyecto
        DB::table('proyectos')->where('id_proyecto', $id)->update([
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion ?? $proyecto->descripcion,
            'updated_at' => now(),
        ]);
        Log::info("Cabecera del proyecto ID {$id} actualizada");

        // 2) Procesar integrantes: upsert + eliminar los que no vienen
        $existingIntegrantes = DB::table('integrantes')->where('id_proyecto', $id)->pluck('id')->toArray();
        $incomingIntegrantes = $request->integrantes ?? [];
        $currentIntegranteIds = [];

        foreach ($incomingIntegrantes as $ing) {
            // Normalizar fechas con Carbon (acepta ISO con Z, sin Z, etc.)
            try {
                $fecha_ingreso_norm = isset($ing['fecha_ingreso']) && !empty($ing['fecha_ingreso'])
                    ? Carbon::parse($ing['fecha_ingreso'])->format('Y-m-d H:i:s')
                    : null;
            } catch (\Exception $ex) {
                $fecha_ingreso_norm = null;
            }

            try {
                $fecha_salida_norm = isset($ing['fecha_salida']) && !empty($ing['fecha_salida'])
                    ? Carbon::parse($ing['fecha_salida'])->format('Y-m-d H:i:s')
                    : null;
            } catch (\Exception $ex) {
                $fecha_salida_norm = null;
            }

            // Si viene id, actualizar esa fila (verificar que pertenezca al proyecto)
            if (!empty($ing['id'])) {
                $row = DB::table('integrantes')->where('id', $ing['id'])->first();
                if (!$row || intval($row->id_proyecto) !== intval($id)) {
                    // intento de modificar fila que no pertenece o no existe
                    Log::warning("Integrante ID {$ing['id']} no pertenece al proyecto {$id} o no existe. Se omite.");
                    continue;
                }

                DB::table('integrantes')->where('id', $ing['id'])->update([
                    'id_user' => $ing['id_user'],
                    'fecha_ingreso' => $fecha_ingreso_norm ?? $row->fecha_ingreso,
                    'fecha_salida' => $fecha_salida_norm ?? $row->fecha_salida,
                    'updated_at' => now(),
                ]);
                $currentIntegranteIds[] = $ing['id'];
                Log::info("  • Integrante ID {$ing['id']} actualizado (user_id: {$ing['id_user']})");
            } else {
                // insertar nuevo integrante (evitar duplicados por id_user)
                $exists = DB::table('integrantes')
                    ->where('id_proyecto', $id)
                    ->where('id_user', $ing['id_user'])
                    ->exists();

                if ($exists) {
                    // obtener ese id existente y mantenerlo
                    $existingId = DB::table('integrantes')
                        ->where('id_proyecto', $id)
                        ->where('id_user', $ing['id_user'])
                        ->value('id');
                    $currentIntegranteIds[] = $existingId;
                    Log::warning("Integrante user_id {$ing['id_user']} ya existe en el proyecto {$id}, se mantiene id {$existingId}");
                    continue;
                }

                $newId = DB::table('integrantes')->insertGetId([
                    'id_proyecto' => $id,
                    'id_user' => $ing['id_user'],
                    'fecha_ingreso' => $fecha_ingreso_norm ?? now(),
                    'fecha_salida' => $fecha_salida_norm ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                $currentIntegranteIds[] = $newId;
                Log::info("  • Nuevo integrante insertado ID: {$newId} (user_id: {$ing['id_user']})");
            }
        }

        // Eliminar integrantes que ya no están en el payload
        $toDelete = array_diff($existingIntegrantes, $currentIntegranteIds);
        if (!empty($toDelete)) {
            DB::table('integrantes')->whereIn('id', $toDelete)->delete();
            Log::info("  • Integrantes eliminados: " . implode(',', $toDelete));
        }

        DB::commit();
        Log::info("Transacción completada correctamente para update proyecto ID: {$id}");

        // Traer proyecto actualizado (con integrantes) y devolverlo para que el frontend confirme
        $proyectoActualizado = DB::table('proyectos')->where('id_proyecto', $id)->first();
        $integrantes = DB::table('integrantes as i')
            ->join('users as u', 'i.id_user', '=', 'u.id_user')
            ->where('i.id_proyecto', $id)
            ->select(
                'i.id',
                'i.id_proyecto',
                'i.id_user',
                'i.fecha_ingreso',
                'i.fecha_salida',
                'i.created_at as integrante_created_at',
                'i.updated_at as integrante_updated_at',
                'u.id_user as user_id',
                'u.nombre_usuario',
                'u.nombre',
                'u.apellido_paterno',
                'u.apellido_materno',
                'u.email',
                'u.telefono'
            )
            ->orderBy('i.fecha_ingreso', 'asc')
            ->get();
        $proyectoActualizado->integrantes = $integrantes;

        return response()->json([
            'message' => 'Proyecto actualizado exitosamente.',
            'proyecto' => $proyectoActualizado
        ], 200);

    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Error al actualizar proyecto, se hizo rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al actualizar proyecto.',
            'details' => $e->getMessage()
        ], 500);
    }
}

}
