<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;

class PublicQuizzesController extends Controller
{
    // Listar quizzes públicos (solo visibles y dentro de fecha si aplica)
    public function index()
    {
        $now = Carbon::now();

        $quizzes = DB::table('quizzes')
            ->where('visible', 1)
            ->where(function($q) use ($now) {
                $q->whereNull('fecha_visible_desde')->orWhere('fecha_visible_desde', '<=', $now);
            })
            ->where(function($q) use ($now) {
                $q->whereNull('fecha_visible_hasta')->orWhere('fecha_visible_hasta', '>=', $now);
            })
            ->select('id_quiz','id_proyecto','titulo','descripcion','tipo','version')
            ->orderByDesc('id_quiz')
            ->get();

        return response()->json(['data' => $quizzes], 200);
    }

    // Mostrar un quiz (con preguntas y opciones). IMPORTANTE: no devolver `valor` por defecto.
    public function show($id)
    {
        $now = Carbon::now();

        $quiz = DB::table('quizzes')
            ->where('id_quiz', $id)
            ->where('visible', 1)
            ->where(function($q) use ($now) {
                $q->whereNull('fecha_visible_desde')->orWhere('fecha_visible_desde', '<=', $now);
            })
            ->where(function($q) use ($now) {
                $q->whereNull('fecha_visible_hasta')->orWhere('fecha_visible_hasta', '>=', $now);
            })
            ->first();

        if (!$quiz) {
            return response()->json(['error' => 'Quiz no encontrado o no visible'], 404);
        }

        $preguntas = DB::table('quiz_questions')
            ->where('id_quiz', $id)
            ->orderBy('orden', 'asc')
            ->get();

        foreach ($preguntas as $p) {
            $opciones = DB::table('question_options')
                ->where('id_pregunta', $p->id_pregunta)
                ->orderBy('orden', 'asc')
                ->get()
                ->map(function($opt) {
                    // No exponer 'valor' al cliente público por defecto.
                    if (isset($opt->valor)) unset($opt->valor);
                    return $opt;
                });

            $p->opciones = $opciones;
        }

        // Devolver reglas (sin puntajes) — para UX pública.
        $reglas = DB::table('quiz_result_rules')
            ->where('id_quiz', $id)
            ->where('activo', 1)
            ->orderBy('prioridad', 'asc')
            ->get()
            ->map(function($r) {
                return [
                    'id_regla' => $r->id_regla,
                    'titulo' => $r->titulo,
                    'descripcion' => $r->descripcion,
                    // 'puntaje_min'/'puntaje_max' intencionalmente omitidos para no exponer rangos públicamente.
                    // 'recursos' u otros campos pueden agregarse si existen en la tabla.
                ];
            });

        return response()->json([
            'quiz' => $quiz,
            'preguntas' => $preguntas,
            'reglas' => $reglas
        ], 200);
    }

    // Submit: recibir respuestas del visitante, calcular puntaje, guardar y devolver resultado
    public function submit(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'fecha_inicio' => 'nullable|date',
            'fecha_fin' => 'nullable|date',
            'answers' => 'required|array|min:1',
            'answers.*.id_pregunta' => 'required|integer|exists:quiz_questions,id_pregunta',
            // cada respuesta puede traer id_opciones (array) o respuesta_text o valor_numerico
            'answers.*.id_opciones' => 'nullable|array',
            'answers.*.id_opciones.*' => 'nullable|integer|exists:question_options,id_opcion',
            'answers.*.respuesta_text' => 'nullable|string',
            'answers.*.valor_numerico' => 'nullable|numeric'
        ]);

        if ($validator->fails()) {
            return response()->json(['error' => 'Datos inválidos', 'details' => $validator->errors()], 422);
        }

        DB::beginTransaction();
        try {
            // 1) Obtener preguntas y opciones con valores (para scoring)
            $preguntasCollection = DB::table('quiz_questions')->where('id_quiz', $id)->get()->keyBy('id_pregunta');
            $preguntas = $preguntasCollection->toArray();
            $opcionesCollection = DB::table('question_options')->whereIn('id_pregunta', array_keys($preguntasCollection->toArray()))->get()->groupBy('id_pregunta');

            // 2) Calculo puntaje
            $total = 0.0;
            $answersPayload = $request->answers;

            foreach ($answersPayload as $ans) {
                $pid = $ans['id_pregunta'];
                $q = $preguntasCollection[$pid] ?? null;
                if (!$q) continue;

                // si vienen id_opciones sumar sus valores
                if (!empty($ans['id_opciones']) && is_array($ans['id_opciones'])) {
                    foreach ($ans['id_opciones'] as $opid) {
                        $opt = DB::table('question_options')->where('id_opcion', $opid)->first();
                        $valor = ($opt && isset($opt->valor) && $opt->valor !== null) ? floatval($opt->valor) : 0.0;
                        $total += $valor;
                    }
                } elseif (isset($ans['valor_numerico']) && $ans['valor_numerico'] !== null) {
                    $total += floatval($ans['valor_numerico']);
                } else {
                    // no hay info de puntaje para esta respuesta -> se suma 0
                }
            }

            // Normalizar total a número (float)
            $total = is_numeric($total) ? floatval($total) : 0.0;
            Log::info("Puntaje calculado para quiz {$id}: {$total}");

            // 3) Guardar quiz_results (puntuacion_total guardada como entero si no hay decimales)
            $storedScore = (floatval($total) == intval($total)) ? intval($total) : $total;

            $idResultado = DB::table('quiz_results')->insertGetId([
                'id_quiz' => $id,
                'id_user' => null, // visitante anónimo
                'puntuacion_total' => $storedScore,
                'resultado_texto' => null, // se actualizará cuando encontremos la regla
                'fecha_inicio' => $request->fecha_inicio ? Carbon::parse($request->fecha_inicio) : now(),
                'fecha_fin' => $request->fecha_fin ? Carbon::parse($request->fecha_fin) : now(),
                'quiz_version' => DB::table('quizzes')->where('id_quiz',$id)->value('version') ?? 1,
                'created_at' => now(),
                'updated_at' => now()
            ]);

            // 4) Insertar answers y answer_options snapshots
            foreach ($answersPayload as $ans) {
                $pid = $ans['id_pregunta'];
                $tipoPregunta = $preguntasCollection[$pid]->tipo_pregunta ?? 'single';

                $tipoRespuesta = 'texto';
                if (!empty($ans['id_opciones']) && is_array($ans['id_opciones'])) {
                    $tipoRespuesta = (count($ans['id_opciones']) > 1) ? 'opcion_multiple' : 'opcion_unica';
                } elseif (isset($ans['valor_numerico']) && $ans['valor_numerico'] !== null) {
                    $tipoRespuesta = 'slider';
                }

                $answerId = DB::table('quiz_answers')->insertGetId([
                    'id_resultado' => $idResultado,
                    'id_pregunta' => $pid,
                    'tipo_respuesta' => $tipoRespuesta,
                    'respuesta_text' => $ans['respuesta_text'] ?? null,
                    'valor_numerico' => $ans['valor_numerico'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now()
                ]);

                // snapshot opciones seleccionadas
                if (!empty($ans['id_opciones']) && is_array($ans['id_opciones'])) {
                    foreach ($ans['id_opciones'] as $opid) {
                        $opt = DB::table('question_options')->where('id_opcion', $opid)->first();
                        DB::table('quiz_answer_options')->insert([
                            'id_answer' => $answerId,
                            'id_opcion' => $opid,
                            'opcion_snapshot_text' => $opt ? $opt->texto_opcion : null,
                            'opcion_snapshot_valor' => $opt ? $opt->valor : null,
                            'created_at' => now()
                        ]);
                    }
                }
            }

            // 5) Encontrar regla que corresponda al total (tipo 'rango' prioritario)
            // Intento directo en DB
            $regla = DB::table('quiz_result_rules')
                ->where('id_quiz', $id)
                ->where('activo', 1)
                ->where('tipo_regla', 'rango')
                ->where('puntaje_min', '<=', $total)
                ->where('puntaje_max', '>=', $total)
                ->orderBy('prioridad', 'asc')
                ->first();

            // Si no la encontramos con la consulta, intentamos buscar en PHP (más tolerante)
            if (!$regla) {
                $todas = DB::table('quiz_result_rules')
                    ->where('id_quiz', $id)
                    ->where('activo', 1)
                    ->where('tipo_regla', 'rango')
                    ->orderBy('prioridad', 'asc')
                    ->get();

                foreach ($todas as $r) {
                    $min = is_numeric($r->puntaje_min) ? floatval($r->puntaje_min) : null;
                    $max = is_numeric($r->puntaje_max) ? floatval($r->puntaje_max) : null;

                    if ($min !== null && $max !== null && $min <= $total && $total <= $max) {
                        $regla = $r;
                        break;
                    }
                }

                // fallback: si aún no hay coincidencia, usar la primera regla activa (evita devolver null)
                if (!$regla && $todas->count()) {
                    $regla = $todas->first();
                    Log::warning("No se encontró regla exacta para puntaje {$total}, usando fallback regla id {$regla->id_regla}");
                }
            }

            $resultadoTitulo = $regla->titulo ?? null;
            $resultadoDesc   = $regla->descripcion ?? null;
            $recursos        = $regla->recursos ?? null; // si existiera columna JSON 'recursos'

            // 6) Actualizar quiz_results con resultado_texto
            DB::table('quiz_results')->where('id_resultado', $idResultado)->update([
                'resultado_texto' => $resultadoTitulo,
                'updated_at' => now()
            ]);

            DB::commit();

            return response()->json([
                'id_resultado' => $idResultado,
                'puntuacion_total' => $storedScore,
                'resultado' => [
                    'titulo' => $resultadoTitulo,
                    'descripcion' => $resultadoDesc,
                    'recursos' => $recursos
                ]
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error en PublicQuizzesController@submit', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json(['error' => 'Error al procesar respuestas.'], 500);
        }
    }
}
