<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Publicacion;
use App\Models\Adjunto;
use App\Models\EstadoPublicacion;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use Illuminate\Support\Facades\Log;

class PublicacionController extends Controller
{
    //-------creacion de publicacion-------
    public function store(Request $request) 
    {
        Log::info('PublicacionController@store - inicio', [
            'ip' => $request->ip(),
            'route' => $request->path(),
            'headers' => [
                'authorization' => $request->header('authorization'),
                'host' => $request->header('host')
            ]
        ]);

        $user = Auth::user();
        if (!$user) {
            Log::warning('PublicacionController@store - usuario no autenticado', [
                'ip' => $request->ip()
            ]);
            return response()->json(['message' => 'Usuario no autenticado'], 401);
        }

        Log::info('PublicacionController@store - usuario autenticado', [
            'user_id' => $user->id_user,
            'nombre_usuario' => optional($user)->nombre ?? null
        ]);

        $validatedData = $request->validate([
            'tipo_publicacion' => 'required|in:infografia,noticia,comunicado,video,pdf,imagen,otro',
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'url' => 'nullable|url',  
            'contenido' => 'nullable|file|mimes:jpeg,png,jpg,pdf', 
            'accion' => 'required|in:guardar,enviar',
            'areas' => 'required|exists:areas,id_area',  
            'tipo_determinante' => 'required|in:Acceso a la Comunidad,Ingreso,Educación,Vivienda y acceso a servicios básicos,Seguridad Alimentaria,Género y Generalidades,Pluriculturalidad,Medio Ambiente y Cambio Climático,Crianza y Salud Mental', 
        ]);

        Log::info('PublicacionController@store - datos validados', [
            'user_id' => $user->id_user,
            'tipo_publicacion' => $validatedData['tipo_publicacion'],
            'titulo' => $validatedData['titulo'],
            'accion' => $validatedData['accion'],
            'areas' => $validatedData['areas'],
            'tipo_determinante' => $validatedData['tipo_determinante'],
        ]);

        //---------Seleccion de estado.----------
        $estadoNombre = $validatedData['accion'] === 'enviar' ? 'pendiente' : 'en_elaboracion';
        $estado = EstadoPublicacion::where('nombre_estado', $estadoNombre)->first();
        if (!$estado) {
            Log::error('PublicacionController@store - estado no encontrado', [
                'estado_busqueda' => $estadoNombre
            ]);
            return response()->json(['message' => 'Estado de publicación no válido'], 400);
        }

        $path = null;

        if ($request->hasFile('contenido')) {
            try {
                $path = $request->file('contenido')->store('publicaciones_contenido', 'public');
                Log::info('PublicacionController@store - archivo guardado', [
                    'user_id' => $user->id_user,
                    'path' => $path
                ]);
            } catch (\Exception $e) {
                Log::error('PublicacionController@store - error al guardar archivo', [
                    'user_id' => $user->id_user,
                    'error' => $e->getMessage()
                ]);
                // No cambiamos la lógica original: si hay error en almacenamiento, podemos seguir o retornar error.
                return response()->json(['message' => 'Error al guardar el contenido'], 500);
            }
        }

        $publicacionData = [
            'id_user' => $user->id_user,
            'tipo_publicacion' => $validatedData['tipo_publicacion'],
            'titulo' => $validatedData['titulo'],
            'descripcion' => $validatedData['descripcion'] ?? null,
            'url' => $validatedData['url'] ?? null, 
            'contenido' => $path, 
            'id_estado_publicacion' => $estado->id_estado_publicacion,
            'fecha_creacion' => now(),
            'tipo_determinante' => $validatedData['tipo_determinante'], 
        ];

        $publicacion = Publicacion::create($publicacionData);

        Log::info('PublicacionController@store - publicacion creada', [
            'id_publicacion' => $publicacion->id_publicacion,
            'user_id' => $publicacion->id_user,
            'id_estado_publicacion' => $publicacion->id_estado_publicacion
        ]);

        try {
            DB::table('area_publicacion')->insert([
                'id_area' => $validatedData['areas'],
                'id_publicacion' => $publicacion->id_publicacion,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            Log::info('PublicacionController@store - area_publicacion insertada', [
                'id_area' => $validatedData['areas'],
                'id_publicacion' => $publicacion->id_publicacion
            ]);
        } catch (\Exception $e) {
            Log::error('PublicacionController@store - error al asignar áreas', [
                'id_publicacion' => $publicacion->id_publicacion,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Error al asignar áreas'], 500);
        }

        try {
            $publicacion->determinantes()->sync([1]); 
            Log::info('PublicacionController@store - determinantes sincronizados', [
                'id_publicacion' => $publicacion->id_publicacion,
                'determinantes' => [1]
            ]);
        } catch (\Exception $e) {
            Log::error('PublicacionController@store - error al asignar determinantes', [
                'id_publicacion' => $publicacion->id_publicacion,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Error al asignar determinantes'], 500);
        }

        Log::info('PublicacionController@store - finalizado correctamente', [
            'id_publicacion' => $publicacion->id_publicacion,
            'user_id' => $user->id_user
        ]);

        return response()->json([
            'message' => 'Publicación creada exitosamente',
            'publicacion' => $publicacion,
            'url_contenido' => $path ? asset("storage/" . $path) : null, 
        ], 201);
    }



    
    
    //-----Actualizar una publicación------
    public function update(Request $request, $id)
    {
        Log::info('PublicacionController@update - inicio', [
            'ip' => $request->ip(),
            'route' => $request->path(),
            'id_param' => $id,
            'authorization' => $request->header('authorization')
        ]);

        $user = Auth::user(); 
        if (!$user) {
            Log::warning('PublicacionController@update - usuario no autenticado', [
                'ip' => $request->ip(),
                'id_param' => $id
            ]);
            return response()->json(['message' => 'Usuario no autenticado'], 401);
        }

        $publicacion = Publicacion::find($id);
        if (!$publicacion) {
            Log::warning('PublicacionController@update - publicacion no encontrada', [
                'id_param' => $id,
                'user_id' => $user->id_user
            ]);
            return response()->json(['message' => 'Publicación no encontrada'], 404);
        }

        $validatedData = $request->validate([
            'titulo' => 'sometimes|required|string|max:255',
            'descripcion' => 'nullable|string',
            'tipo_publicacion' => 'sometimes|required|in:infografia,noticia,comunicado,video,pdf,imagen,otro',
            'accion' => 'required|in:guardar,enviar',
            'areas' => 'required|exists:areas,id_area',  
            'tipo_determinante' => 'sometimes|required|in:Acceso a la Comunidad,Ingreso,Educación,Vivienda y acceso a servicios básicos,Seguridad Alimentaria,Género y Generalidades,Pluriculturalidad,Medio Ambiente y Cambio Climático,Crianza y Salud Mental',
        ]);

        Log::info('PublicacionController@update - datos validados', [
            'id_publicacion' => $publicacion->id_publicacion,
            'user_id' => $user->id_user,
            'validated' => [
                'titulo' => $validatedData['titulo'] ?? null,
                'tipo_publicacion' => $validatedData['tipo_publicacion'] ?? null,
                'accion' => $validatedData['accion'],
                'areas' => $validatedData['areas'],
                'tipo_determinante' => $validatedData['tipo_determinante'] ?? null
            ]
        ]);

        $estadoNombre = $validatedData['accion'] === 'enviar' ? 'pendiente' : 'en_elaboracion';
        $estado = EstadoPublicacion::where('nombre_estado', $estadoNombre)->first();
        if (!$estado) {
            Log::error('PublicacionController@update - estado no valido', [
                'estado_busqueda' => $estadoNombre,
                'id_publicacion' => $publicacion->id_publicacion
            ]);
            return response()->json(['message' => 'Estado de publicación no válido'], 400);
        }

        $publicacion->titulo = $validatedData['titulo'] ?? $publicacion->titulo;
        $publicacion->descripcion = $validatedData['descripcion'] ?? $publicacion->descripcion;
        $publicacion->tipo_publicacion = $validatedData['tipo_publicacion'] ?? $publicacion->tipo_publicacion;
        $publicacion->id_estado_publicacion = $estado->id_estado_publicacion;
        $publicacion->tipo_determinante = $validatedData['tipo_determinante'] ?? $publicacion->tipo_determinante;
        $publicacion->fecha_actualizacion = now(); 

        $publicacion->save();

        Log::info('PublicacionController@update - publicacion actualizada', [
            'id_publicacion' => $publicacion->id_publicacion,
            'id_estado_publicacion' => $publicacion->id_estado_publicacion
        ]);

        try {
            DB::table('area_publicacion')->where('id_publicacion', $publicacion->id_publicacion)->delete();
        
            DB::table('area_publicacion')->insert([
                'id_area' => $validatedData['areas'], 
                'id_publicacion' => $publicacion->id_publicacion,
                'updated_at' => now()
            ]);

            Log::info('PublicacionController@update - area_publicacion actualizada', [
                'id_area' => $validatedData['areas'],
                'id_publicacion' => $publicacion->id_publicacion
            ]);
        } catch (\Exception $e) {
            Log::error('PublicacionController@update - error al actualizar áreas', [
                'id_publicacion' => $publicacion->id_publicacion,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Error al actualizar áreas'], 500);
        }
        
        try {
            $publicacion->determinantes()->sync([1]); 
            Log::info('PublicacionController@update - determinantes sincronizados', [
                'id_publicacion' => $publicacion->id_publicacion,
                'determinantes' => [1]
            ]);
        } catch (\Exception $e) {
            Log::error('PublicacionController@update - error al actualizar determinantes', [
                'id_publicacion' => $publicacion->id_publicacion,
                'error' => $e->getMessage()
            ]);
            return response()->json(['message' => 'Error al actualizar determinantes'], 500);
        }

        Log::info('PublicacionController@update - finalizado correctamente', [
            'id_publicacion' => $publicacion->id_publicacion,
            'user_id' => $user->id_user
        ]);

        return response()->json([
            'message' => 'Publicación actualizada exitosamente',
            'publicacion' => $publicacion
        ], 200);
    }







    

//-----Mostrar una publicación por ID------
    public function show($id)
    {
        Log::info('PublicacionController@show - inicio', [
            'id_param' => $id
        ]);

        $publicacion = Publicacion::findOrFail($id);

        $areas = DB::table('area_publicacion')
            ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
            ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
            ->pluck('areas.nombre_area');

       
        $determinantes = $publicacion->determinantes->pluck('id_determinante');

        Log::info('PublicacionController@show - publicacion cargada', [
            'id_publicacion' => $publicacion->id_publicacion,
            'tipo_publicacion' => $publicacion->tipo_publicacion,
            'areas_count' => $areas->count()
        ]);
        
        $response = [
            'id_publicacion' => $publicacion->id_publicacion,
            'tipo_publicacion' => $publicacion->tipo_publicacion,
            'tipo_determinante'=> $publicacion->tipo_determinante,
            'titulo' => $publicacion->titulo,
            'contenido' => $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null,
            'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
            'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
            'url' => $publicacion->url ?? '',
            'descripcion' => $publicacion->descripcion ?? '',
            'estado_publicacion' => $publicacion->estado->nombre_estado,
            'areas' => $areas, 
            //'determinantes' => $determinantes,
        ];

        Log::info('PublicacionController@show - respuesta preparada', [
            'id_publicacion' => $publicacion->id_publicacion
        ]);

        return response()->json($response);
    }


 

    

    
 //-----Listar todas las publicaciones por id.-----
        public function index()
        {
            $userId = auth()->id(); 

            Log::info('PublicacionController@index - inicio', [
                'auth_user_id' => $userId
            ]);
        
            $publicaciones = Publicacion::where('id_user', $userId)
                                ->orderBy('id_publicacion', 'desc')
                                ->get();
        
            Log::info('PublicacionController@index - publicaciones obtenidas', [
                'auth_user_id' => $userId,
                'count' => $publicaciones->count()
            ]);

            $publicaciones = $publicaciones->map(function ($publicacion) {
             
                $estado = $publicacion->estado->nombre_estado; 
        
                $urlContenido = $publicacion->contenido ? asset("storage/" . $publicacion->contenido) : null;

                $areas = DB::table('area_publicacion')
                    ->join('areas', 'area_publicacion.id_area', '=', 'areas.id_area')
                    ->where('area_publicacion.id_publicacion', $publicacion->id_publicacion)
                    ->pluck('areas.nombre_area');
        
                //$determinantes = $publicacion->determinantes->pluck('id_determinante');
        
                return [
                    'id_publicacion' => $publicacion->id_publicacion,
                    'tipo_publicacion' => $publicacion->tipo_publicacion,
                    'titulo' => $publicacion->titulo,
                    'contenido' => $urlContenido,
                    'fecha_creacion' => Carbon::parse($publicacion->fecha_creacion)->format('Y-m-d H:i:s'),
                    'fecha_aprobacion' => $publicacion->fecha_aprobacion ? Carbon::parse($publicacion->fecha_aprobacion)->format('Y-m-d H:i:s') : null,
                    'url' => $publicacion->url ?? '',
                    'descripcion' => $publicacion->descripcion ?? '',
                    'estado_publicacion' => $estado,
                    'areas' => $areas,
                    //'determinantes' => $determinantes,
                    'tipo_determinante' => $publicacion->tipo_determinante, 
                ];
            });
        
            Log::info('PublicacionController@index - respuesta preparada', [
                'auth_user_id' => $userId,
                'result_count' => $publicaciones->count()
            ]);
        
            return response()->json($publicaciones);
        }
 

 //--------listado de pyublicaciones pendientes-----
    public function PendientesTotal() 
    {
        Log::info('PublicacionController@PendientesTotal - inicio');

        $estadoPendiente = EstadoPublicacion::where('nombre_estado', 'pendiente')->first();

        if (!$estadoPendiente) {
            Log::error('PublicacionController@PendientesTotal - estado pendiente no encontrado');
            return response()->json(['message' => 'Estado pendiente no encontrado'], 404);
        }

        $totalPendientes = Publicacion::where('id_estado_publicacion', $estadoPendiente->id_estado_publicacion)->count();

        Log::info('PublicacionController@PendientesTotal - total calculado', [
            'estado_id' => $estadoPendiente->id_estado_publicacion,
            'total' => $totalPendientes
        ]);

        return response()->json([
            'total_pendientes' => $totalPendientes,
        ], 200);
    }




    // Método auxiliar para obtener la URL del contenido basado en su tipo
    protected function getContenidoUrl($publicacion)
    {
        $tipoPublicacion = $publicacion->tipo_publicacion;
        
        switch ($tipoPublicacion) {
            case 'pdf':
            case 'imagen':
            case 'infografia':
            case 'noticia':
            case 'comunicado':
            case 'otro':
                
                return asset('storage/' . $publicacion->contenido);
        
            case 'video':
        
                return $publicacion->url;
        
            default:
                return null;
        }
    }
}
