<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class QuizesController extends Controller
{
    /**
     * Listar quizzes con sus preguntas, opciones y reglas.
     */
    public function index()
{
    try {
        // 1) Obtener quizzes (joins para traer datos relacionados como usuario y proyecto)
        $quizzes = DB::table('quizzes')
            ->leftJoin('users', 'quizzes.id_user', '=', 'users.id_user')
            ->leftJoin('proyectos', 'quizzes.id_proyecto', '=', 'proyectos.id_proyecto')
            ->select(
                'quizzes.id_quiz',
                'quizzes.id_proyecto',
                'quizzes.id_user',
                'quizzes.titulo',
                'quizzes.descripcion',
                'quizzes.tipo',
                'quizzes.version',
                'quizzes.visible',
                'quizzes.fecha_visible_desde',
                'quizzes.fecha_visible_hasta',
                'quizzes.created_at as quiz_created_at',
                'quizzes.updated_at as quiz_updated_at',
                'users.nombre_usuario as usuario_nombre',
                'proyectos.nombre as proyecto_nombre'
            )
            ->orderByDesc('quizzes.id_quiz')
            ->get();

        // 2) Mapear cada quiz para agregar sus preguntas, opciones y reglas (estilo index2)
        $quizzesConDatos = $quizzes->map(function ($q) {
            // 2.1 Preguntas del quiz (todos los campos)
            $preguntas = DB::table('quiz_questions')
                ->select('quiz_questions.*')
                ->where('id_quiz', $q->id_quiz)
                ->orderBy('orden', 'asc')
                ->get();

            // 2.2 Para cada pregunta, obtener sus opciones (todos los campos)
            $preguntasConOpciones = $preguntas->map(function ($p) {
                $opciones = DB::table('question_options')
                    ->select('question_options.*')
                    ->where('id_pregunta', $p->id_pregunta)
                    ->orderBy('orden', 'asc')
                    ->get();

                // adjuntar opciones a la pregunta
                $p->opciones = $opciones;
                return $p;
            });

            // 2.3 Reglas de resultado del quiz (todos los campos)
            $reglas = DB::table('quiz_result_rules')
                ->select('quiz_result_rules.*')
                ->where('id_quiz', $q->id_quiz)
                ->orderBy('prioridad', 'asc')
                ->get();

            // 2.4 Adjuntar al objeto quiz
            $q->preguntas = $preguntasConOpciones;
            $q->reglas = $reglas;

            // (Opcional) contar preguntas / opciones si lo necesitas en el frontend
            $q->cantidad_preguntas = $preguntasConOpciones->count();
            // ejemplo: contar opciones totales
            $q->cantidad_opciones = $preguntasConOpciones->reduce(function ($carry, $item) {
                return $carry + ($item->opciones->count() ?? 0);
            }, 0);

            return $q;
        });

        return response()->json([
            'message' => 'Quizzes listados correctamente.',
            'data' => $quizzesConDatos
        ], 200);

    } catch (\Exception $e) {
        Log::error('Error al listar quizzes (index estilo index2)', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al listar quizzes.',
            'details' => $e->getMessage()
        ], 500);
    }
}


    public function store(Request $request)
    {
        Log::info('Inicio de store() en QuizesController', ['input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'id_proyecto' => 'required|integer|exists:proyectos,id_proyecto',
            'id_user' => 'nullable|integer|exists:users,id_user',
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'tipo' => 'required|in:puntaje,condicional,mixto',
            'version' => 'nullable|integer|min:1',
            'visible' => 'nullable|boolean',
            'fecha_visible_desde' => 'nullable|date',
            'fecha_visible_hasta' => 'nullable|date|after_or_equal:fecha_visible_desde',
            'preguntas' => 'required|array|min:1',
            'preguntas.*.orden' => 'required|integer|min:0',
            'preguntas.*.texto_pregunta' => 'required|string',
            'preguntas.*.tipo_pregunta' => 'required|in:single,multiple,open_text,numeric,date,likert,slider,boolean',
            'preguntas.*.ayuda' => 'nullable|string|max:255',
            'preguntas.*.opciones' => 'nullable|array',
            'preguntas.*.opciones.*.texto_opcion' => 'required_with:preguntas.*.opciones|string',
            'preguntas.*.opciones.*.etiqueta' => 'nullable|string|max:100',
            'preguntas.*.opciones.*.valor' => 'nullable|numeric',
            'preguntas.*.opciones.*.orden' => 'nullable|integer|min:0',
            'reglas' => 'nullable|array',
            'reglas.*.tipo_regla' => 'required_with:reglas|in:rango,condicional',
            'reglas.*.condicion' => 'nullable|string|max:500',
            'reglas.*.puntaje_min' => 'nullable|integer|min:0',
            'reglas.*.puntaje_max' => 'nullable|integer|min:0',
            'reglas.*.titulo' => 'nullable|string|max:100',
            'reglas.*.descripcion' => 'nullable|string',
            'reglas.*.prioridad' => 'nullable|integer|min:0',
            'reglas.*.activo' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al crear quiz', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        Log::info('Transacción iniciada para crear quiz');

        try {
            $idQuiz = DB::table('quizzes')->insertGetId([
                'id_proyecto' => $request->id_proyecto,
                'id_user' => $request->id_user,
                'titulo' => $request->titulo,
                'descripcion' => $request->descripcion,
                'tipo' => $request->tipo,
                'version' => $request->version ?? 1,
                'visible' => $request->visible ?? 0,
                'fecha_visible_desde' => $request->fecha_visible_desde,
                'fecha_visible_hasta' => $request->fecha_visible_hasta,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            Log::info("Quiz creado con ID: {$idQuiz}");

            // Insertar preguntas y opciones
            $preguntas = $request->preguntas;
            foreach ($preguntas as $preg) {
                $idPregunta = DB::table('quiz_questions')->insertGetId([
                    'id_quiz' => $idQuiz,
                    'orden' => $preg['orden'],
                    'texto_pregunta' => $preg['texto_pregunta'],
                    'tipo_pregunta' => $preg['tipo_pregunta'],
                    'ayuda' => $preg['ayuda'] ?? null,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

                Log::info(" - Pregunta creada ID: {$idPregunta} para quiz {$idQuiz}");

                if (!empty($preg['opciones']) && is_array($preg['opciones'])) {
                    foreach ($preg['opciones'] as $opt) {
                        DB::table('question_options')->insert([
                            'id_pregunta' => $idPregunta,
                            'texto_opcion' => $opt['texto_opcion'],
                            'etiqueta' => $opt['etiqueta'] ?? null,
                            'valor' => isset($opt['valor']) ? $opt['valor'] : null,
                            'orden' => $opt['orden'] ?? 0,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                    }
                    Log::info("   • Opciones insertadas para pregunta {$idPregunta}");
                }
            }

            // Insertar reglas (si existen)
            $reglas = $request->reglas ?? [];
            foreach ($reglas as $reg) {
                DB::table('quiz_result_rules')->insert([
                    'id_quiz' => $idQuiz,
                    'tipo_regla' => $reg['tipo_regla'],
                    'condicion' => $reg['condicion'] ?? null,
                    'puntaje_min' => $reg['puntaje_min'] ?? null,
                    'puntaje_max' => $reg['puntaje_max'] ?? null,
                    'titulo' => $reg['titulo'] ?? ($reg['resultado_title'] ?? null),
                    'descripcion' => $reg['descripcion'] ?? ($reg['resultado_texto'] ?? null),
                    'prioridad' => $reg['prioridad'] ?? 0,
                    'activo' => $reg['activo'] ?? 1,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            DB::commit();
            Log::info("Transacción finalizada correctamente. Quiz ID: {$idQuiz}");

            return response()->json([
                'message' => 'Quiz creado exitosamente.',
                'id_quiz' => $idQuiz
            ], 201);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al crear quiz, se hizo rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al crear quiz.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar un quiz (cabecera + preguntas/opciones + reglas).
     *
     * Para preguntas/opciones: si incluye id_pregunta / id_opcion se actualiza, si no existe se inserta.
     * Las preguntas/opciones/reglas que ya existían y no vienen en el request se eliminan.
     */
    public function update(Request $request, $id)
    {
        Log::info('Inicio de update() en QuizesController', ['id' => $id, 'input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'id_proyecto' => 'required|integer|exists:proyectos,id_proyecto',
            'id_user' => 'nullable|integer|exists:users,id_user',
            'titulo' => 'required|string|max:255',
            'descripcion' => 'nullable|string',
            'tipo' => 'required|in:puntaje,condicional,mixto',
            'version' => 'nullable|integer|min:1',
            'visible' => 'nullable|boolean',
            'fecha_visible_desde' => 'nullable|date',
            'fecha_visible_hasta' => 'nullable|date|after_or_equal:fecha_visible_desde',
            'preguntas' => 'required|array|min:1',
            'preguntas.*.id_pregunta' => 'nullable|integer|exists:quiz_questions,id_pregunta',
            'preguntas.*.orden' => 'required|integer|min:0',
            'preguntas.*.texto_pregunta' => 'required|string',
            'preguntas.*.tipo_pregunta' => 'required|in:single,multiple,open_text,numeric,date,likert,slider,boolean',
            'preguntas.*.ayuda' => 'nullable|string|max:255',
            'preguntas.*.opciones' => 'nullable|array',
            'preguntas.*.opciones.*.id_opcion' => 'nullable|integer|exists:question_options,id_opcion',
            'preguntas.*.opciones.*.texto_opcion' => 'required_with:preguntas.*.opciones|string',
            'preguntas.*.opciones.*.etiqueta' => 'nullable|string|max:100',
            'preguntas.*.opciones.*.valor' => 'nullable|numeric',
            'preguntas.*.opciones.*.orden' => 'nullable|integer|min:0',
            'reglas' => 'nullable|array',
            'reglas.*.id_regla' => 'nullable|integer|exists:quiz_result_rules,id_regla',
            'reglas.*.tipo_regla' => 'required_with:reglas|in:rango,condicional',
            'reglas.*.condicion' => 'nullable|string|max:500',
            'reglas.*.puntaje_min' => 'nullable|integer|min:0',
            'reglas.*.puntaje_max' => 'nullable|integer|min:0',
            'reglas.*.titulo' => 'nullable|string|max:100',
            'reglas.*.descripcion' => 'nullable|string',
            'reglas.*.prioridad' => 'nullable|integer|min:0',
            'reglas.*.activo' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al actualizar quiz', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        Log::info("Transacción iniciada para update quiz ID: {$id}");

        try {
            $quiz = DB::table('quizzes')->where('id_quiz', $id)->first();
            if (!$quiz) {
                Log::error("Quiz no encontrado ID: {$id}");
                return response()->json(['error' => "Quiz {$id} no existe."], 404);
            }

            // 1) Actualizar cabecera del quiz
            DB::table('quizzes')->where('id_quiz', $id)->update([
                'id_proyecto' => $request->id_proyecto,
                'id_user' => $request->id_user,
                'titulo' => $request->titulo,
                'descripcion' => $request->descripcion,
                'tipo' => $request->tipo,
                'version' => $request->version ?? $quiz->version,
                'visible' => $request->visible ?? $quiz->visible,
                'fecha_visible_desde' => $request->fecha_visible_desde,
                'fecha_visible_hasta' => $request->fecha_visible_hasta,
                'updated_at' => now(),
            ]);
            Log::info("Cabecera del quiz ID {$id} actualizada");

            // 2) Procesar preguntas y opciones (upsert + borrar las que faltan)
            $existingPreguntas = DB::table('quiz_questions')->where('id_quiz', $id)->pluck('id_pregunta')->toArray();
            $incomingPreguntas = $request->preguntas;

            $currentPreguntaIds = [];

            foreach ($incomingPreguntas as $preg) {
                if (!empty($preg['id_pregunta'])) {
                    // actualizar pregunta existente
                    DB::table('quiz_questions')->where('id_pregunta', $preg['id_pregunta'])->update([
                        'orden' => $preg['orden'],
                        'texto_pregunta' => $preg['texto_pregunta'],
                        'tipo_pregunta' => $preg['tipo_pregunta'],
                        'ayuda' => $preg['ayuda'] ?? null,
                        'updated_at' => now(),
                    ]);
                    $idPregunta = $preg['id_pregunta'];
                    Log::info(" - Pregunta actualizada ID: {$idPregunta}");
                } else {
                    // insertar nueva pregunta
                    $idPregunta = DB::table('quiz_questions')->insertGetId([
                        'id_quiz' => $id,
                        'orden' => $preg['orden'],
                        'texto_pregunta' => $preg['texto_pregunta'],
                        'tipo_pregunta' => $preg['tipo_pregunta'],
                        'ayuda' => $preg['ayuda'] ?? null,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                    Log::info(" - Nueva pregunta creada ID: {$idPregunta}");
                }

                $currentPreguntaIds[] = $idPregunta;

                // Opciones de la pregunta
                $existingOpciones = DB::table('question_options')->where('id_pregunta', $idPregunta)->pluck('id_opcion')->toArray();
                $incomingOpciones = $preg['opciones'] ?? [];
                $currentOpcionIds = [];

                foreach ($incomingOpciones as $opt) {
                    if (!empty($opt['id_opcion'])) {
                        // actualizar opción existente
                        DB::table('question_options')->where('id_opcion', $opt['id_opcion'])->update([
                            'texto_opcion' => $opt['texto_opcion'],
                            'etiqueta' => $opt['etiqueta'] ?? null,
                            'valor' => isset($opt['valor']) ? $opt['valor'] : null,
                            'orden' => $opt['orden'] ?? 0,
                            'updated_at' => now(),
                        ]);
                        $idOpcion = $opt['id_opcion'];
                        Log::info("   • Opción actualizada ID: {$idOpcion} (pregunta {$idPregunta})");
                    } else {
                        // insertar nueva opción
                        $idOpcion = DB::table('question_options')->insertGetId([
                            'id_pregunta' => $idPregunta,
                            'texto_opcion' => $opt['texto_opcion'],
                            'etiqueta' => $opt['etiqueta'] ?? null,
                            'valor' => isset($opt['valor']) ? $opt['valor'] : null,
                            'orden' => $opt['orden'] ?? 0,
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);
                        Log::info("   • Nueva opción creada ID: {$idOpcion} (pregunta {$idPregunta})");
                    }

                    $currentOpcionIds[] = $idOpcion;
                }

                // Eliminar opciones que ya no vienen
                $opcionesToDelete = array_diff($existingOpciones, $currentOpcionIds);
                if (!empty($opcionesToDelete)) {
                    DB::table('question_options')->whereIn('id_opcion', $opcionesToDelete)->delete();
                    Log::info("   • Opciones eliminadas: " . implode(',', $opcionesToDelete) . " (pregunta {$idPregunta})");
                }
            }

            // Eliminar preguntas que ya no vienen
            $preguntasToDelete = array_diff($existingPreguntas, $currentPreguntaIds);
            if (!empty($preguntasToDelete)) {
                // Al borrar preguntas, las opciones y demás se eliminarán por FK cascade
                DB::table('quiz_questions')->whereIn('id_pregunta', $preguntasToDelete)->delete();
                Log::info(" - Preguntas eliminadas: " . implode(',', $preguntasToDelete));
            }

            // 3) Procesar reglas (upsert + eliminar las que faltan)
            $existingReglas = DB::table('quiz_result_rules')->where('id_quiz', $id)->pluck('id_regla')->toArray();
            $incomingReglas = $request->reglas ?? [];
            $currentReglaIds = [];

            foreach ($incomingReglas as $reg) {
                if (!empty($reg['id_regla'])) {
                    DB::table('quiz_result_rules')->where('id_regla', $reg['id_regla'])->update([
                        'tipo_regla' => $reg['tipo_regla'],
                        'condicion' => $reg['condicion'] ?? null,
                        'puntaje_min' => $reg['puntaje_min'] ?? null,
                        'puntaje_max' => $reg['puntaje_max'] ?? null,
                        'titulo' => $reg['titulo'] ?? ($reg['resultado_title'] ?? null),
                        'descripcion' => $reg['descripcion'] ?? ($reg['resultado_texto'] ?? null),
                        'prioridad' => $reg['prioridad'] ?? 0,
                        'activo' => $reg['activo'] ?? 1,
                        'updated_at' => now(),
                    ]);
                    $idRegla = $reg['id_regla'];
                    Log::info(" - Regla actualizada ID: {$idRegla}");
                } else {
                    $idRegla = DB::table('quiz_result_rules')->insertGetId([
                        'id_quiz' => $id,
                        'tipo_regla' => $reg['tipo_regla'],
                        'condicion' => $reg['condicion'] ?? null,
                        'puntaje_min' => $reg['puntaje_min'] ?? null,
                        'puntaje_max' => $reg['puntaje_max'] ?? null,
                        'titulo' => $reg['titulo'] ?? ($reg['resultado_title'] ?? null),
                        'descripcion' => $reg['descripcion'] ?? ($reg['resultado_texto'] ?? null),
                        'prioridad' => $reg['prioridad'] ?? 0,
                        'activo' => $reg['activo'] ?? 1,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                    Log::info(" - Nueva regla creada ID: {$idRegla}");
                }

                $currentReglaIds[] = $idRegla;
            }

            // Eliminar reglas que no vienen
            $reglasToDelete = array_diff($existingReglas, $currentReglaIds);
            if (!empty($reglasToDelete)) {
                DB::table('quiz_result_rules')->whereIn('id_regla', $reglasToDelete)->delete();
                Log::info(" - Reglas eliminadas: " . implode(',', $reglasToDelete));
            }

            DB::commit();
            Log::info("Transacción completada correctamente para update quiz ID: {$id}");

            return response()->json([
                'message' => 'Quiz actualizado exitosamente.',
                'id_quiz' => $id
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar quiz, se hizo rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al actualizar quiz.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
