<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class RecursosController extends Controller
{


public function index(Request $request)
    {
        try {
            // Construimos la query base (mis mismos joins que tenías)
            $query = DB::table('recursos')

                ->leftJoin('categorias', 'recursos.id_categoria', '=', 'categorias.id_categoria')
                ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
                ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
                ->leftJoin('estados', 'recursos.id_estado', '=', 'estados.id_estado')

                // creador y su rol
                ->leftJoin('users as creator', 'recursos.created_by', '=', 'creator.id_user')
                ->leftJoin('roles as creator_role', 'creator.id_rol', '=', 'creator_role.id_rol')

                // aprobador y su rol
                ->leftJoin('users as approver', 'recursos.approved_by', '=', 'approver.id_user')
                ->leftJoin('roles as approver_role', 'approver.id_rol', '=', 'approver_role.id_rol')

                // intentamos enlazar la fila de 'integrantes' correspondiente al creador dentro del proyecto
                ->leftJoin('integrantes as ing', function ($join) {
                    $join->on('ing.id_user', '=', 'recursos.created_by')
                         ->on('ing.id_proyecto', '=', 'proyectos.id_proyecto')
                         ->whereNull('ing.fecha_salida');
                });

            // ---------------------------
            // Reglas de acceso por rol
            // ---------------------------
            if (!auth()->check()) {
                return response()->json([
                    'error' => 'No autenticado',
                    'details' => 'Se requiere autenticación para listar recursos según el rol del usuario.'
                ], 401);
            }

            $authUser = auth()->user();

            // Obtener nombre del rol del usuario autenticado (seguro si en users guardas id_rol)
            $authRoleNombre = DB::table('roles')->where('id_rol', $authUser->id_rol)->value('nombre_rol');
            $authRoleLower = strtolower($authRoleNombre ?? '');

            // Detectar si es admin (tus roles son 'SuperAdmin' y 'Usuario')
            $isAdmin = ($authRoleLower === 'superadmin');

            // Si viene el parámetro mine=1 forzamos recursos propios (útil si un admin quiere ver sólo los suyos)
            $mineParam = $request->query('mine');

            // Si viene created_by y el solicitante es admin permitimos filtrar por ese created_by
            $createdByParam = $request->query('created_by');

            if ($mineParam) {
                // forzar recursos propios
                $query->where('recursos.created_by', $authUser->id_user ?? $authUser->id ?? auth()->id());
            } elseif (!empty($createdByParam) && $isAdmin) {
                // admin solicita recursos de un usuario concreto
                $query->where('recursos.created_by', $createdByParam);
            } elseif ($isAdmin) {
                // SuperAdmin: por defecto puede ver todos los recursos (no se aplica filtro)
                // (no hacemos where)
            } else {
                // usuario normal: sólo sus recursos
                $query->where('recursos.created_by', $authUser->id_user ?? $authUser->id ?? auth()->id());
            }

            // Selección de columnas (igual que antes)
            $recursos = $query->select(
                    'recursos.id_recurso',
                    'proyectos.id_proyecto as id_proyecto',
                    'secciones.id_seccion as id_seccion',
                    'categorias.id_categoria as id_categoria',
                    'recursos.titulo',
                    'recursos.descripcion',
                    'recursos.url',
                    'recursos.portada',
                    'recursos.created_by',
                    'ing.id as id_integrante',
                    'ing.fecha_ingreso as integrante_fecha_ingreso',
                    'recursos.id_estado',
                    'estados.nombre_estado as estado_nombre',
                    'recursos.approved_by',
                    'recursos.approved_at',
                    'categorias.nombre as categoria_nombre',
                    'secciones.nombre as seccion_nombre',
                    'proyectos.nombre as proyecto_nombre',
                    'creator.id_user as creator_id_user',
                    DB::raw("CONCAT(creator.nombre, ' ', COALESCE(creator.apellido_paterno,''), ' ', COALESCE(creator.apellido_materno,'')) as created_by_fullname"),
                    'creator.nombre_usuario as created_by_nombre_usuario',
                    'creator.email as created_by_email',
                    'creator.telefono as created_by_telefono',
                    'creator_role.nombre_rol as creator_role_nombre',
                    'approver.id_user as approver_id_user',
                    DB::raw("CONCAT(approver.nombre, ' ', COALESCE(approver.apellido_paterno,''), ' ', COALESCE(approver.apellido_materno,'')) as approved_by_fullname"),
                    DB::raw("(SELECT GROUP_CONCAT(DISTINCT tipo_archivo SEPARATOR ',') FROM recurso_archivos WHERE id_recurso = recursos.id_recurso) as tipos_archivos"),

                    'approver.nombre_usuario as approved_by_nombre_usuario',
                    'approver.email as approved_by_email',
                    'approver_role.nombre_rol as approver_role_nombre',
                    'recursos.created_at',
                    'recursos.updated_at'
                )
                ->orderBy('recursos.created_at', 'desc')
                ->get();

            $recursosConArchivos = $recursos->map(function ($r) {
                $archivos = DB::table('recurso_archivos')
                    ->select('id', 'id_recurso', 'ruta', 'nombre_original', 'tipo_archivo', 'orden', 'created_at', 'updated_at')
                    ->where('id_recurso', $r->id_recurso)
                    ->orderBy('orden', 'asc')
                    ->get();

                $r->archivos = $archivos;
                return $r;
            });

            return response()->json([
                'message' => 'Recursos listados correctamente.',
                'data' => $recursosConArchivos
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al listar recursos', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener recursos',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function store(Request $request)
{
    Log::info('Inicio de store() en RecursosController', ['input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'id_proyecto' => 'required|integer|exists:proyectos,id_proyecto',
        'id_seccion' => 'nullable|integer|exists:secciones,id_seccion',
        'id_categoria' => 'nullable|integer|exists:categorias,id_categoria',
        'created_by' => 'required|integer|exists:users,id_user',
        'titulo' => 'required|string|max:255|unique:recursos,titulo',
        'descripcion' => 'nullable|string|max:1000',
        'url' => 'nullable|url|max:1000',
        'archivos' => 'nullable|array',
        'archivos.*.file' => 'nullable|file|mimes:jpg,jpeg,png,webp,gif,pdf,mp4|max:10240',
        'archivos.*.ruta' => 'nullable|string|max:1000',
        'archivos.*.nombre_original' => 'nullable|string|max:255',
        'archivos.*.tipo_archivo' => 'required_with:archivos|in:imagen,documento,video,otro,url',
        'archivos.*.orden' => 'nullable|integer',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al crear recurso', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors(),
        ], 422);
    }

    // Verificar que created_by es integrante del proyecto indicado
    $isIntegrante = DB::table('integrantes')
        ->where('id_proyecto', $request->id_proyecto)
        ->where('id_user', $request->created_by)
        ->exists();

    if (! $isIntegrante) {
        Log::warning('Intento de crear recurso por usuario no integrante', [
            'id_proyecto' => $request->id_proyecto,
            'created_by' => $request->created_by
        ]);

        return response()->json([
            'error' => 'Permiso denegado. El usuario no es integrante del proyecto seleccionado.'
        ], 403);
    }

    DB::beginTransaction();
    Log::info('Transacción iniciada para crear recurso');

    $storedFiles = [];
    $createdDirs = [];

    try {
        $integranteRecord = DB::table('integrantes')
            ->where('id_proyecto', $request->id_proyecto)
            ->where('id_user', $request->created_by)
            ->first();
        $id_integrante = $integranteRecord->id ?? null;

        $idRecurso = DB::table('recursos')->insertGetId([
            'id_proyecto' => $request->id_proyecto,
            'id_seccion' => $request->id_seccion ?? null,
            'id_categoria' => $request->id_categoria ?? null,
            'titulo' => $request->titulo,
            'descripcion' => $request->descripcion ?? null,
            'url' => $request->url ?? null,
            'portada' => $request->portada ?? null,
            'id_estado' => $request->id_estado ?? 1,
            'created_by' => $request->created_by,
            'id_integrante' => $id_integrante,
            'approved_by' => $request->approved_by ?? null,
            'approved_at' => $request->approved_at ?? null,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        Log::info("Recurso creado con ID: {$idRecurso}");

        // Construir ruta de carpeta del recurso según jerarquía:
        // proyectos/{id_proyecto}/secciones/{id_seccion?}/categorias/{id_categoria?}/recursos/{id_recurso}
        $parts = [];
        $parts[] = 'proyectos';
        $parts[] = $request->id_proyecto;
        if (!empty($request->id_seccion)) {
            $parts[] = 'secciones';
            $parts[] = $request->id_seccion;
        }
        if (!empty($request->id_categoria)) {
            $parts[] = 'categorias';
            $parts[] = $request->id_categoria;
        }
        $parts[] = 'recursos';
        $parts[] = $idRecurso;

        $resourceDir = implode('/', $parts); // ruta relativa en disk 'public'

        // Crear directorio del recurso (makeDirectory es idempotente y crea recursivamente)
        \Illuminate\Support\Facades\Storage::disk('public')->makeDirectory($resourceDir);
        $createdDirs[] = $resourceDir;
        Log::info("Carpeta de recurso creada: {$resourceDir}");

        $archivosInput = $request->input('archivos', []);

        foreach ($archivosInput as $idx => $a) {
            $file = $request->file("archivos.$idx.file");

            if ($file instanceof \Illuminate\Http\UploadedFile) {
                // Guardar dentro de la carpeta del recurso
                $path = $file->store($resourceDir, 'public');
                $rutaPublica = $this->toAbsolutePublicUrl($path);
                $nombreOriginal = $file->getClientOriginalName();
                $storedFiles[] = $path;
                Log::info("Archivo subido: {$path}");
            } elseif (!empty($a['ruta'])) {
                // Si vienen rutas externas ya almacenadas o URLs relativas en disco
                $rutaPublica = $this->toAbsolutePublicUrl($a['ruta']);
                $nombreOriginal = $a['nombre_original'] ?? null;
                Log::info("Archivo referenciado por ruta: {$a['ruta']}");
            } else {
                throw new \Exception("Cada elemento de 'archivos' debe incluir 'file' (archivo) o 'ruta' (string). Faltante en índice {$idx}.");
            }

            DB::table('recurso_archivos')->insert([
                'id_recurso' => $idRecurso,
                'ruta' => $rutaPublica,
                'nombre_original' => $nombreOriginal ?? null,
                'tipo_archivo' => $a['tipo_archivo'],
                'orden' => $a['orden'] ?? $idx,
                'created_at' => now(),
                'updated_at' => now(),
            ]);
        }

        DB::commit();

        $query = DB::table('recursos')
            ->leftJoin('categorias', 'recursos.id_categoria', '=', 'categorias.id_categoria')
            ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->leftJoin('estados', 'recursos.id_estado', '=', 'estados.id_estado')
            ->leftJoin('users as creator', 'recursos.created_by', '=', 'creator.id_user')
            ->leftJoin('roles as creator_role', 'creator.id_rol', '=', 'creator_role.id_rol')
            ->leftJoin('users as approver', 'recursos.approved_by', '=', 'approver.id_user')
            ->leftJoin('roles as approver_role', 'approver.id_rol', '=', 'approver_role.id_rol')
            ->leftJoin('integrantes as ing', function ($join) {
                $join->on('ing.id_user', '=', 'recursos.created_by')
                     ->on('ing.id_proyecto', '=', 'proyectos.id_proyecto')
                     ->whereNull('ing.fecha_salida');
            });

        $recurso = $query->select(
                'recursos.id_recurso',
                'proyectos.id_proyecto as id_proyecto',
                'secciones.id_seccion as id_seccion',
                'categorias.id_categoria as id_categoria',
                'recursos.titulo',
                'recursos.descripcion',
                'recursos.url',
                'recursos.portada',
                'recursos.created_by',
                'ing.id as id_integrante',
                'ing.fecha_ingreso as integrante_fecha_ingreso',
                'recursos.id_estado',
                'estados.nombre_estado as estado_nombre',
                'recursos.approved_by',
                'recursos.approved_at',
                'categorias.nombre as categoria_nombre',
                'secciones.nombre as seccion_nombre',
                'proyectos.nombre as proyecto_nombre',
                'creator.id_user as creator_id_user',
                DB::raw("CONCAT(creator.nombre, ' ', COALESCE(creator.apellido_paterno,''), ' ', COALESCE(creator.apellido_materno,'')) as created_by_fullname"),
                'creator.nombre_usuario as created_by_nombre_usuario',
                'creator.email as created_by_email',
                'creator.telefono as created_by_telefono',
                'creator_role.nombre_rol as creator_role_nombre',
                'approver.id_user as approver_id_user',
                DB::raw("CONCAT(approver.nombre, ' ', COALESCE(approver.apellido_paterno,''), ' ', COALESCE(approver.apellido_materno,'')) as approved_by_fullname"),
                'approver.nombre_usuario as approved_by_nombre_usuario',
                'approver.email as approved_by_email',
                'approver_role.nombre_rol as approver_role_nombre',
                'recursos.created_at',
                'recursos.updated_at'
            )
            ->where('recursos.id_recurso', $idRecurso)
            ->orderBy('recursos.created_at', 'desc')
            ->first();

        // Adjuntamos archivos (misma estructura que en index)
        $recurso->archivos = DB::table('recurso_archivos')
            ->where('id_recurso', $idRecurso)
            ->orderBy('orden', 'asc')
            ->get();

        Log::info("Transacción finalizada para recurso ID: {$idRecurso}");

        return response()->json([
            'message' => 'Recurso creado exitosamente.',
            'data' => $recurso,
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();

        // Eliminar archivos subidos físicamente
        foreach ($storedFiles as $p) {
            try {
                \Illuminate\Support\Facades\Storage::disk('public')->delete($p);
                Log::info("Archivo físico eliminado por rollback: {$p}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar archivo en rollback: {$p}. Error: " . $ex->getMessage());
            }
        }

        // Eliminar directorios creados (reverse order por seguridad)
        if (!empty($createdDirs)) {
            $createdDirs = array_reverse($createdDirs);
            foreach ($createdDirs as $dir) {
                try {
                    \Illuminate\Support\Facades\Storage::disk('public')->deleteDirectory($dir);
                    Log::info("Carpeta eliminada por rollback: {$dir}");
                } catch (\Exception $ex) {
                    Log::warning("No se pudo eliminar la carpeta en rollback: {$dir}. Error: " . $ex->getMessage());
                }
            }
        }

        Log::error('Error al crear recurso, se hizo rollback', ['exception_message' => $e->getMessage(), 'stack' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al crear recurso.',
            'details' => $e->getMessage(),
        ], 500);
    }
}

    public function update(Request $request, $id)
{
    Log::info('Inicio de update() en RecursosController', ['id' => $id, 'input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'id_proyecto' => 'nullable|integer|exists:proyectos,id_proyecto',
        'id_seccion' => 'nullable|integer|exists:secciones,id_seccion',
        'id_categoria' => 'nullable|integer|exists:categorias,id_categoria',
        'created_by' => 'required|integer|exists:users,id_user',
        'titulo' => "required|string|max:255|unique:recursos,titulo,{$id},id_recurso",

        'descripcion' => 'nullable|string|max:1000',
        'url' => 'nullable|url|max:1000',
        'archivos' => 'nullable|array',
        'archivos.*.id' => 'nullable|integer|min:1',
        'archivos.*.file' => 'nullable|file|mimes:jpg,jpeg,png,webp,gif,pdf,mp4|max:10240',
        'archivos.*.ruta' => 'nullable|string|max:1000',
        'archivos.*.nombre_original' => 'nullable|string|max:255',
        'archivos.*.tipo_archivo' => 'required_with:archivos|in:imagen,documento,video,otro,url',
        'archivos.*.orden' => 'nullable|integer',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al actualizar recurso', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors(),
        ], 422);
    }

    DB::beginTransaction();
    Log::info('Transacción iniciada para actualizar recurso ID: '.$id);

    $storedFiles = [];
    $oldFilesToDelete = [];

    try {
        $recurso = DB::table('recursos')->where('id_recurso', $id)->first();
        if (! $recurso) {
            Log::error("Recurso no encontrado ID: {$id}");
            return response()->json(['error' => "Recurso {$id} no existe."], 404);
        }

        $targetProyecto = $request->filled('id_proyecto') ? $request->id_proyecto : $recurso->id_proyecto;

        // Verificar que el actor (created_by) es integrante del proyecto objetivo
        $isIntegrante = DB::table('integrantes')
            ->where('id_proyecto', $targetProyecto)
            ->where('id_user', $request->created_by)
            ->exists();

        if (! $isIntegrante) {
            DB::rollBack();
            Log::warning('Intento de actualizar recurso por usuario no integrante', [
                'id_recurso' => $id,
                'target_proyecto' => $targetProyecto,
                'actor' => $request->created_by
            ]);

            return response()->json([
                'error' => 'Permiso denegado. El usuario que intenta actualizar no es integrante del proyecto objetivo.'
            ], 403);
        }

        DB::table('recursos')->where('id_recurso', $id)->update([
            'id_proyecto' => $targetProyecto,
            'id_seccion' => $request->id_seccion ?? null,
            'id_categoria' => $request->id_categoria ?? null,
            'titulo' => $request->titulo,
            'descripcion' => $request->descripcion ?? null,
            'url' => $request->url ?? null,
            'portada' => $request->portada ?? null,
            'id_estado' => $request->id_estado ?? $recurso->id_estado ?? 1,
            'updated_at' => now(),
        ]);
        Log::info("Cabecera recurso ID {$id} actualizada");

        // Obtener archivos existentes para detectar borrados y rutas antiguas
        $existingArchivos = DB::table('recurso_archivos')
            ->where('id_recurso', $id)
            ->get();
        $existingIds = $existingArchivos->pluck('id')->toArray();
        $existingMap = $existingArchivos->pluck('ruta', 'id')->toArray();

        $currentArchivoIds = [];
        $archivosInput = $request->input('archivos', []);

        foreach ($archivosInput as $idx => $a) {
            $file = $request->file("archivos.$idx.file");

            if (!empty($a['id'])) {
                // actualizar registro existente
                $rutaPublica = null;
                $nombreOriginal = $a['nombre_original'] ?? null;

                if ($file instanceof UploadedFile) {
                    // guardar nuevo archivo
                    $path = $file->store("recursos/{$id}", 'public'); // relativo
                    $rutaPublica = $this->toAbsolutePublicUrl($path);
                    $storedFiles[] = $path;

                    // programar eliminación del fichero antiguo (si existía)
                    if (!empty($existingMap[$a['id']])) {
                        $old = $existingMap[$a['id']];
                        $oldRelative = $this->getRelativePathFromStoredRuta($old);
                        if ($oldRelative && $oldRelative !== $path) {
                            $oldFilesToDelete[] = $oldRelative;
                        }
                    }
                } else {
                    // si no trae file, conservar la ruta que venga o la existente
                    if (!empty($a['ruta'])) {
                        $rutaPublica = $this->toAbsolutePublicUrl($a['ruta']);
                    } else {
                        $rutaPublica = $existingMap[$a['id']] ?? null;
                    }
                }

                DB::table('recurso_archivos')
                    ->where('id', $a['id'])
                    ->update([
                        'ruta' => $rutaPublica,
                        'nombre_original' => $nombreOriginal,
                        'tipo_archivo' => $a['tipo_archivo'],
                        'orden' => $a['orden'] ?? $idx,
                        'updated_at' => now(),
                    ]);
                $currentArchivoIds[] = $a['id'];
                Log::info(" - Archivo actualizado ID: {$a['id']}");
            } else {
                // insertar nuevo archivo
                $rutaPublica = null;
                $nombreOriginal = $a['nombre_original'] ?? null;

                if ($file instanceof UploadedFile) {
                    $path = $file->store("recursos/{$id}", 'public');
                    $rutaPublica = $this->toAbsolutePublicUrl($path);
                    $nombreOriginal = $file->getClientOriginalName();
                    $storedFiles[] = $path;
                } else {
                    // si viene solo ruta, convertir si es relativa local
                    $rutaPublica = !empty($a['ruta']) ? $this->toAbsolutePublicUrl($a['ruta']) : null;
                }

                $newId = DB::table('recurso_archivos')->insertGetId([
                    'id_recurso' => $id,
                    'ruta' => $rutaPublica,
                    'nombre_original' => $nombreOriginal ?? null,
                    'tipo_archivo' => $a['tipo_archivo'],
                    'orden' => $a['orden'] ?? $idx,
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
                $currentArchivoIds[] = $newId;
                Log::info(" - Nuevo archivo creado ID: {$newId}");
            }
        }

        // Eliminar filas de archivos que ya no vienen en la petición (pero borrar fichero físico solo tras commit)
        $toDelete = array_diff($existingIds, $currentArchivoIds);
        if (!empty($toDelete)) {
            $rutasParaBorrar = DB::table('recurso_archivos')->whereIn('id', $toDelete)->pluck('ruta')->toArray();
            foreach ($rutasParaBorrar as $ruta) {
                if ($ruta) {
                    $rel = $this->getRelativePathFromStoredRuta($ruta);
                    if ($rel) $oldFilesToDelete[] = $rel;
                }
            }

            DB::table('recurso_archivos')->whereIn('id', $toDelete)->delete();
            Log::info(' - Filas de archivos eliminadas: ' . implode(',', $toDelete));
        }

        DB::commit();

        // Después del commit: eliminar físicamente los ficheros antiguos programados
        foreach (array_unique($oldFilesToDelete) as $rel) {
            try {
                Storage::disk('public')->delete($rel);
                Log::info("Archivo físico antiguo eliminado después del commit: {$rel}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar archivo físico {$rel} después del commit: " . $ex->getMessage());
            }
        }

        // --- NUEVO: Recuperar el recurso actualizado con los mismos joins/selección que index() ---
        $recursoActualizado = DB::table('recursos')
            ->leftJoin('categorias', 'recursos.id_categoria', '=', 'categorias.id_categoria')
            ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->leftJoin('estados', 'recursos.id_estado', '=', 'estados.id_estado')
            ->leftJoin('users as creator', 'recursos.created_by', '=', 'creator.id_user')
            ->leftJoin('roles as creator_role', 'creator.id_rol', '=', 'creator_role.id_rol')
            ->leftJoin('users as approver', 'recursos.approved_by', '=', 'approver.id_user')
            ->leftJoin('roles as approver_role', 'approver.id_rol', '=', 'approver_role.id_rol')
            ->leftJoin('integrantes as ing', function ($join) {
                $join->on('ing.id_user', '=', 'recursos.created_by')
                     ->on('ing.id_proyecto', '=', 'proyectos.id_proyecto')
                     ->whereNull('ing.fecha_salida');
            })
            ->select(
                'recursos.id_recurso',
                'proyectos.id_proyecto as id_proyecto',
                'secciones.id_seccion as id_seccion',
                'categorias.id_categoria as id_categoria',
                'recursos.titulo',
                'recursos.descripcion',
                'recursos.url',
                'recursos.portada',
                'recursos.created_by',
                'ing.id as id_integrante',
                'ing.fecha_ingreso as integrante_fecha_ingreso',
                'recursos.id_estado',
                'estados.nombre_estado as estado_nombre',
                'recursos.approved_by',
                'recursos.approved_at',
                'categorias.nombre as categoria_nombre',
                'secciones.nombre as seccion_nombre',
                'proyectos.nombre as proyecto_nombre',
                'creator.id_user as creator_id_user',
                DB::raw("CONCAT(creator.nombre, ' ', COALESCE(creator.apellido_paterno,''), ' ', COALESCE(creator.apellido_materno,'')) as created_by_fullname"),
                'creator.nombre_usuario as created_by_nombre_usuario',
                'creator.email as created_by_email',
                'creator.telefono as created_by_telefono',
                'creator_role.nombre_rol as creator_role_nombre',
                'approver.id_user as approver_id_user',
                DB::raw("CONCAT(approver.nombre, ' ', COALESCE(approver.apellido_paterno,''), ' ', COALESCE(approver.apellido_materno,'')) as approved_by_fullname"),
                'approver.nombre_usuario as approved_by_nombre_usuario',
                'approver.email as approved_by_email',
                'approver_role.nombre_rol as approver_role_nombre',
                'recursos.created_at',
                'recursos.updated_at'
            )
            ->where('recursos.id_recurso', $id)
            ->first();

        // Adjuntar archivos (con select explícito que incluye tipo_archivo)
        $recursoActualizado->archivos = DB::table('recurso_archivos')
            ->select('id', 'id_recurso', 'ruta', 'nombre_original', 'tipo_archivo', 'orden', 'created_at', 'updated_at')
            ->where('id_recurso', $id)
            ->orderBy('orden', 'asc')
            ->get();

        Log::info("Transacción finalizada correctamente para recurso ID: {$id}");

        return response()->json([
            'message' => 'Recurso actualizado exitosamente.',
            'data' => $recursoActualizado,
        ], 200);
    } catch (\Exception $e) {
        DB::rollBack();

        // Eliminar archivos nuevos que fueron almacenados si hubo rollback
        foreach ($storedFiles as $p) {
            try {
                Storage::disk('public')->delete($p);
                Log::info("Archivo físico eliminado por rollback (update): {$p}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar archivo en rollback (update): {$p}. Error: " . $ex->getMessage());
            }
        }

        Log::error('Error al actualizar recurso, se hizo rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al actualizar recurso.',
            'details' => $e->getMessage(),
        ], 500);
    }
}

    private function toAbsolutePublicUrl(string $candidate): string
    {

        if (preg_match('/^https?:\\/\\//i', $candidate)) {
            return $candidate;
        }

        if (strpos($candidate, '/storage/') !== false) {
            $relative = ltrim(substr($candidate, strpos($candidate, '/storage/') + strlen('/storage/')), '/');
            return url('storage/' . $relative);
        }

        if (strpos($candidate, 'storage/') === 0) {
            $relative = ltrim(substr($candidate, strlen('storage/')), '/');
            return url('storage/' . $relative);
        }

        return url('storage/' . ltrim($candidate, '/'));
    }


    private function getRelativePathFromStoredRuta(string $ruta)
    {
        if (empty($ruta)) return null;

        if (preg_match('/^https?:\\/\\//i', $ruta)) {
            $pos = strpos($ruta, '/storage/');
            if ($pos !== false) {
                return ltrim(substr($ruta, $pos + strlen('/storage/')), '/');
            }

            return null;
        }

        if (strpos($ruta, '/storage/') === 0) {
            return ltrim(substr($ruta, strlen('/storage/')), '/');
        }
        if (strpos($ruta, 'storage/') === 0) {
            return ltrim(substr($ruta, strlen('storage/')), '/');
        }

        if (!preg_match('/^https?:\\/\\//i', $ruta)) {
            return ltrim($ruta, '/');
        }

        return null;
    }


public function revision(Request $request, $id)
{
    Log::info('Inicio revision() en RecursosController', ['id_recurso' => $id, 'input' => $request->all()]);

    // validación base
    $validator = Validator::make($request->all(), [
        'id_estado' => 'required|integer|exists:estados,id_estado',
        'comentarios' => 'nullable|string',
        'portada' => 'nullable', // puede ser file o string (ruta)
    ]);

    if ($validator->fails()) {
        Log::warning('Validación base fallida en revision()', ['errors' => $validator->errors()->all()]);
        return response()->json(['error' => 'Datos inválidos.', 'details' => $validator->errors()], 422);
    }

    $user = auth()->user();
    if (! $user) {
        return response()->json(['error' => 'No autorizado.'], 401);
    }
    $userId = $user->id_user;

    DB::beginTransaction();

    $oldPortadaToDelete = null;
    $newStoredPath = null;

    try {
        $recurso = DB::table('recursos')->where('id_recurso', $id)->first();
        if (! $recurso) {
            DB::rollBack();
            return response()->json(['error' => "Recurso {$id} no encontrado."], 404);
        }

        // obtener nombre del estado para reglas condicionales
        $estadoNombre = DB::table('estados')->where('id_estado', $request->id_estado)->value('nombre_estado');
        $estadoNorm = strtolower(trim((string)$estadoNombre));

        // reglas condicionales
        if ($estadoNorm === 'rechazado') {
            if (! $request->filled('comentarios')) {
                DB::rollBack();
                return response()->json(['error' => 'Validación: cuando se rechaza, los comentarios son obligatorios.'], 422);
            }
        }

        if ($estadoNorm === 'aprobado') {
            // si viene archivo como portada, validar tipo/tamaño
            $hasFilePortada = $request->hasFile('portada') && $request->file('portada') instanceof UploadedFile;
            if ($hasFilePortada) {
                $file = $request->file('portada');
                $fileValidator = Validator::make(['portada' => $file], [
                    'portada' => 'file|mimes:jpg,jpeg,png,webp,gif|max:5120' // 5MB
                ]);
                if ($fileValidator->fails()) {
                    DB::rollBack();
                    return response()->json(['error' => 'Portada inválida.', 'details' => $fileValidator->errors()], 422);
                }
            }
        }

        // Insertar revisión (auditoría)
        $idRevision = DB::table('revisiones')->insertGetId([
            'id_recurso' => $id,
            'id_user' => $userId,
            'id_estado' => $request->id_estado,
            'comentarios' => $request->comentarios ?? null,
            'fecha_revision' => now(),
        ]);

        // Preparar actualización del recurso
        $updateData = [
            'id_estado' => $request->id_estado,
            'updated_at' => now(),
        ];

        if ($estadoNorm === 'aprobado') {
            $updateData['approved_by'] = $userId;
            $updateData['approved_at'] = now();

            // manejar portada: si viene file -> guardarlo en storage/app/public/portadas/{id}
            if ($request->hasFile('portada') && $request->file('portada') instanceof UploadedFile) {
                $file = $request->file('portada');
                $path = $file->store("portadas/{$id}", 'public'); // ej: "portadas/123/portada.jpg"
                $newStoredPath = $path;
                $updateData['portada'] = $this->toAbsolutePublicUrl($path);

                // programar eliminación de la portada antigua si era manejable por Storage::disk('public')
                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel && $oldRel !== $path) $oldPortadaToDelete = $oldRel;
                }
            } elseif ($request->filled('portada')) {
                // portada enviada como ruta string (no movemos el fichero)
                $updateData['portada'] = $this->toAbsolutePublicUrl($request->portada);

                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel) $oldPortadaToDelete = $oldRel;
                }
            }
        } else {
            // en otros estados no cambiamos approved_by/approved_at
            // $updateData['approved_by'] = null;
            // $updateData['approved_at'] = null;
        }

        DB::table('recursos')->where('id_recurso', $id)->update($updateData);

        DB::commit();

        // borrar portada antigua DESPUÉS del commit (seguro)
        if (!empty($oldPortadaToDelete)) {
            try {
                Storage::disk('public')->delete($oldPortadaToDelete);
                Log::info("Portada antigua eliminada: {$oldPortadaToDelete}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar portada antigua {$oldPortadaToDelete}: " . $ex->getMessage());
            }
        }

        $revision = DB::table('revisiones')->where('id_revision', $idRevision)->first();

        $query = DB::table('recursos')
            ->leftJoin('categorias', 'recursos.id_categoria', '=', 'categorias.id_categoria')
            ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->leftJoin('estados', 'recursos.id_estado', '=', 'estados.id_estado')
            ->leftJoin('users as creator', 'recursos.created_by', '=', 'creator.id_user')
            ->leftJoin('roles as creator_role', 'creator.id_rol', '=', 'creator_role.id_rol')
            ->leftJoin('users as approver', 'recursos.approved_by', '=', 'approver.id_user')
            ->leftJoin('roles as approver_role', 'approver.id_rol', '=', 'approver_role.id_rol')
            ->leftJoin('integrantes as ing', function ($join) {
                $join->on('ing.id_user', '=', 'recursos.created_by')
                     ->on('ing.id_proyecto', '=', 'proyectos.id_proyecto')
                     ->whereNull('ing.fecha_salida');
            });

        $recurso = $query->select(
                'recursos.id_recurso',
                'proyectos.id_proyecto as id_proyecto',
                'secciones.id_seccion as id_seccion',
                'categorias.id_categoria as id_categoria',
                'recursos.titulo',
                'recursos.descripcion',
                'recursos.url',
                'recursos.portada',
                'recursos.created_by',
                'ing.id as id_integrante',
                'ing.fecha_ingreso as integrante_fecha_ingreso',
                'recursos.id_estado',
                'estados.nombre_estado as estado_nombre',
                'recursos.approved_by',
                'recursos.approved_at',
                'categorias.nombre as categoria_nombre',
                'secciones.nombre as seccion_nombre',
                'proyectos.nombre as proyecto_nombre',
                'creator.id_user as creator_id_user',
                DB::raw("CONCAT(creator.nombre, ' ', COALESCE(creator.apellido_paterno,''), ' ', COALESCE(creator.apellido_materno,'')) as created_by_fullname"),
                'creator.nombre_usuario as created_by_nombre_usuario',
                'creator.email as created_by_email',
                'creator.telefono as created_by_telefono',
                'creator_role.nombre_rol as creator_role_nombre',
                'approver.id_user as approver_id_user',
                DB::raw("CONCAT(approver.nombre, ' ', COALESCE(approver.apellido_paterno,''), ' ', COALESCE(approver.apellido_materno,'')) as approved_by_fullname"),
                'approver.nombre_usuario as approved_by_nombre_usuario',
                'approver.email as approved_by_email',
                'approver_role.nombre_rol as approver_role_nombre',
                'recursos.created_at',
                'recursos.updated_at'
            )
            ->where('recursos.id_recurso', $id)
            ->orderBy('recursos.created_at', 'desc')
            ->first();

        if ($recurso) {
            $recurso->archivos = DB::table('recurso_archivos')
                ->where('id_recurso', $id)
                ->orderBy('orden', 'asc')
                ->get();
        }

        Log::info("Revisión registrada id_revision={$idRevision} recurso={$id} por user={$userId} estado={$request->id_estado}");

        return response()->json([
            'message' => 'Revisión registrada correctamente.',
            'revision' => $revision,
            'data' => $recurso
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();

        // si guardamos un nuevo archivo y hubo error, borrarlo aquí
        if (!empty($newStoredPath)) {
            try { Storage::disk('public')->delete($newStoredPath); } catch (\Exception $ex) {}
        }

        Log::error('Error en revision(): ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
        return response()->json(['error' => 'Error al procesar la revisión.', 'details' => $e->getMessage()], 500);
    }
}

public function updateRevision(Request $request, $id)
{
    // Funcionalmente similar a revision() pero actualiza la revisión existente del mismo user si existe
    Log::info('Inicio updateRevision() en RecursosController', ['id_recurso' => $id, 'input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'id_estado' => 'required|integer|exists:estados,id_estado',
        'comentarios' => 'nullable|string',
        'portada' => 'nullable', // puede ser file o string (ruta)
    ]);

    if ($validator->fails()) {
        Log::warning('Validación base fallida en updateRevision()', ['errors' => $validator->errors()->all()]);
        return response()->json(['error' => 'Datos inválidos.', 'details' => $validator->errors()], 422);
    }

    $user = auth()->user();
    if (! $user) {
        return response()->json(['error' => 'No autorizado.'], 401);
    }
    $userId = $user->id_user;

    DB::beginTransaction();

    $oldPortadaToDelete = null;
    $newStoredPath = null;

    try {
        $recurso = DB::table('recursos')->where('id_recurso', $id)->first();
        if (! $recurso) {
            DB::rollBack();
            return response()->json(['error' => "Recurso {$id} no encontrado."], 404);
        }

        // obtener nombre del estado para reglas condicionales
        $estadoNombre = DB::table('estados')->where('id_estado', $request->id_estado)->value('nombre_estado');
        $estadoNorm = strtolower(trim((string)$estadoNombre));

        // reglas condicionales: si se quiere marcar como rechazado, comentarios obligatorios
        if ($estadoNorm === 'rechazado') {
            if (! $request->filled('comentarios')) {
                DB::rollBack();
                return response()->json(['error' => 'Validación: cuando se rechaza, los comentarios son obligatorios.'], 422);
            }
        }

        // si nueva acción es aprobado y viene portada -> validar (igual que en revision())
        $hasFilePortada = $request->hasFile('portada') && $request->file('portada') instanceof \Illuminate\Http\UploadedFile;
        if ($estadoNorm === 'aprobado' && $hasFilePortada) {
            $file = $request->file('portada');
            $fileValidator = Validator::make(['portada' => $file], [
                'portada' => 'file|mimes:jpg,jpeg,png,webp,gif|max:5120' // 5MB
            ]);
            if ($fileValidator->fails()) {
                DB::rollBack();
                return response()->json(['error' => 'Portada inválida.', 'details' => $fileValidator->errors()], 422);
            }
        }

        // Buscar la revisión más reciente hecha por este usuario para este recurso
        $ultimaRevision = DB::table('revisiones')
            ->where('id_recurso', $id)
            ->where('id_user', $userId)
            ->orderBy('fecha_revision', 'desc')
            ->first();

        if ($ultimaRevision) {
            // actualizamos la revisión existente
            $updateRevData = [
                'id_estado' => $request->id_estado,
                'comentarios' => $request->comentarios ?? $ultimaRevision->comentarios,
                'fecha_revision' => now(),
            ];

            DB::table('revisiones')->where('id_revision', $ultimaRevision->id_revision)->update($updateRevData);
            $idRevision = $ultimaRevision->id_revision;
        } else {
            // inserto nueva revisión
            $idRevision = DB::table('revisiones')->insertGetId([
                'id_recurso' => $id,
                'id_user' => $userId,
                'id_estado' => $request->id_estado,
                'comentarios' => $request->comentarios ?? null,
                'fecha_revision' => now(),
            ]);
        }

        // Preparar actualización del recurso (manejo approved_by/approved_at)
        $updateData = [
            'id_estado' => $request->id_estado,
            'updated_at' => now(),
        ];

        // Si se aprueba -> set approved_by/approved_at
        if ($estadoNorm === 'aprobado') {
            $updateData['approved_by'] = $userId;
            $updateData['approved_at'] = now();

            if ($hasFilePortada) {
                $file = $request->file('portada');
                $path = $file->store("portadas/{$id}", 'public');
                $newStoredPath = $path;
                $updateData['portada'] = $this->toAbsolutePublicUrl($path);

                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel && $oldRel !== $path) $oldPortadaToDelete = $oldRel;
                }
            } elseif ($request->filled('portada')) {
                // portada enviada como ruta string
                $updateData['portada'] = $this->toAbsolutePublicUrl($request->portada);

                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel) $oldPortadaToDelete = $oldRel;
                }
            }
        } else {
            // Si se deja de estar en aprobado, limpiamos approved_by/approved_at
            // (solo si estaba aprobado antes)
            $prevEstadoNombre = DB::table('estados')->where('id_estado', $recurso->id_estado)->value('nombre_estado');
            if (strtolower((string)$prevEstadoNombre) === 'aprobado') {
                $updateData['approved_by'] = null;
                $updateData['approved_at'] = null;
            }
        }

        DB::table('recursos')->where('id_recurso', $id)->update($updateData);

        DB::commit();

        // borrar portada antigua DESPUÉS del commit (si programada)
        if (!empty($oldPortadaToDelete)) {
            try {
                Storage::disk('public')->delete($oldPortadaToDelete);
                Log::info("Portada antigua eliminada: {$oldPortadaToDelete}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar portada antigua {$oldPortadaToDelete}: " . $ex->getMessage());
            }
        }

        // recuperar revision actualizada e info del recurso (misma estructura que revision())
        $revision = DB::table('revisiones')->where('id_revision', $idRevision)->first();

        $query = DB::table('recursos')
            ->leftJoin('categorias', 'recursos.id_categoria', '=', 'categorias.id_categoria')
            ->leftJoin('secciones', 'categorias.id_seccion', '=', 'secciones.id_seccion')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->leftJoin('estados', 'recursos.id_estado', '=', 'estados.id_estado')
            ->leftJoin('users as creator', 'recursos.created_by', '=', 'creator.id_user')
            ->leftJoin('roles as creator_role', 'creator.id_rol', '=', 'creator_role.id_rol')
            ->leftJoin('users as approver', 'recursos.approved_by', '=', 'approver.id_user')
            ->leftJoin('roles as approver_role', 'approver.id_rol', '=', 'approver_role.id_rol')
            ->leftJoin('integrantes as ing', function ($join) {
                $join->on('ing.id_user', '=', 'recursos.created_by')
                     ->on('ing.id_proyecto', '=', 'proyectos.id_proyecto')
                     ->whereNull('ing.fecha_salida');
            });

        $recursoActualizado = $query->select(
                'recursos.id_recurso',
                'proyectos.id_proyecto as id_proyecto',
                'secciones.id_seccion as id_seccion',
                'categorias.id_categoria as id_categoria',
                'recursos.titulo',
                'recursos.descripcion',
                'recursos.url',
                'recursos.portada',
                'recursos.created_by',
                'ing.id as id_integrante',
                'ing.fecha_ingreso as integrante_fecha_ingreso',
                'recursos.id_estado',
                'estados.nombre_estado as estado_nombre',
                'recursos.approved_by',
                'recursos.approved_at',
                'categorias.nombre as categoria_nombre',
                'secciones.nombre as seccion_nombre',
                'proyectos.nombre as proyecto_nombre',
                'creator.id_user as creator_id_user',
                DB::raw("CONCAT(creator.nombre, ' ', COALESCE(creator.apellido_paterno,''), ' ', COALESCE(creator.apellido_materno,'')) as created_by_fullname"),
                'creator.nombre_usuario as created_by_nombre_usuario',
                'creator.email as created_by_email',
                'creator.telefono as created_by_telefono',
                'creator_role.nombre_rol as creator_role_nombre',
                'approver.id_user as approver_id_user',
                DB::raw("CONCAT(approver.nombre, ' ', COALESCE(approver.apellido_paterno,''), ' ', COALESCE(approver.apellido_materno,'')) as approved_by_fullname"),
                'approver.nombre_usuario as approved_by_nombre_usuario',
                'approver.email as approved_by_email',
                'approver_role.nombre_rol as approver_role_nombre',
                'recursos.created_at',
                'recursos.updated_at'
            )
            ->where('recursos.id_recurso', $id)
            ->first();

        if ($recursoActualizado) {
            $recursoActualizado->archivos = DB::table('recurso_archivos')
                ->where('id_recurso', $id)
                ->orderBy('orden', 'asc')
                ->get();
        }

        Log::info("updateRevision finalizada id_revision={$idRevision} recurso={$id} por user={$userId} estado={$request->id_estado}");

        return response()->json([
            'message' => 'Revisión actualizada correctamente.',
            'revision' => $revision,
            'data' => $recursoActualizado
        ], 200);

    } catch (\Exception $e) {
        DB::rollBack();

        // si guardamos un nuevo archivo y hubo error, borrarlo aquí
        if (!empty($newStoredPath)) {
            try { Storage::disk('public')->delete($newStoredPath); } catch (\Exception $ex) {}
        }

        Log::error('Error en updateRevision(): ' . $e->getMessage(), ['trace' => $e->getTraceAsString()]);
        return response()->json(['error' => 'Error al procesar la actualización de la revisión.', 'details' => $e->getMessage()], 500);
    }
}


}
