<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class RecursosPublicController extends Controller
{
    public function index(Request $request)
    {
        try {
            $projectKey = trim($request->query('project', 'sanamente'));
            $sectionKey = trim($request->query('section', 'consejos'));

            // Buscar proyecto por nombre (preferido) o por nombre (case-insensitive)
            $project = DB::table('proyectos')
                ->whereRaw('LOWER(nombre) = ?', [strtolower($projectKey)])
                ->orWhereRaw('LOWER(nombre) = ?', [strtolower($projectKey)])
                ->first();

            if (! $project) {
                // Si no existe el proyecto, devolvemos vacíos pero status 200 (público)
                return response()->json([
                    'message' => "Proyecto '{$projectKey}' no encontrado.",
                    'data' => [
                        'proyecto' => null,
                        'seccion' => null,
                        'categorias' => [],
                        'recursos' => []
                    ]
                ], 200);
            }

            // Buscar sección dentro del proyecto por nombre o nombre (case-insensitive)
            $seccion = DB::table('secciones')
                ->where('id_proyecto', $project->id_proyecto)
                ->where(function ($q) use ($sectionKey) {
                    $q->whereRaw('LOWER(nombre) = ?', [strtolower($sectionKey)])
                      ->orWhereRaw('LOWER(nombre) = ?', [strtolower($sectionKey)]);
                })
                ->first();

            if (! $seccion) {
                return response()->json([
                    'message' => "Sección '{$sectionKey}' no encontrada en el proyecto '{$projectKey}'.",
                    'data' => [
                        'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                        'seccion' => null,
                        'categorias' => [],
                        'recursos' => []
                    ]
                ], 200);
            }

            // Obtener categorías de la sección (solo campos necesarios)
            $categorias = DB::table('categorias')
                ->where('id_seccion', $seccion->id_seccion)
                ->select('id_categoria', 'nombre', 'nombre')
                ->orderBy('nombre')
                ->get();

            // Determinar id del estado "aprobado" (buscar por nombre 'aprobado' case-insensitive)
            $approvedEstadoId = DB::table('estados')
                ->whereRaw('LOWER(nombre_estado) = ?', ['aprobado'])
                ->value('id_estado');

            // Si no encontramos el estado "aprobado", intentamos usar id = 2 como fallback (opcional)
            if (empty($approvedEstadoId)) {
                $approvedEstadoId = 2; // ajusta si en tu BD es otro id o maneja como error
            }

            // Obtener recursos aprobados del proyecto y sección, seleccionar solo campos necesarios
            $recursos = DB::table('recursos')
                ->where('id_proyecto', $project->id_proyecto)
                ->where('id_seccion', $seccion->id_seccion)
                ->where('id_estado', $approvedEstadoId)
                ->select(
                    'id_recurso',
                    'id_proyecto',
                    'id_seccion',
                    'id_categoria',
                    'titulo',
                    'descripcion',
                    'portada',
                    'url',
                    'created_at'
                )
                ->orderBy('created_at', 'desc')
                ->get();

            if ($recursos->isEmpty()) {
                return response()->json([
                    'message' => 'No hay recursos aprobados para esta sección/proyecto.',
                    'data' => [
                        'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                        'seccion' => ['id_seccion' => $seccion->id_seccion, 'nombre' => $seccion->nombre, 'nombre' => $seccion->nombre ?? null],
                        'categorias' => $categorias,
                        'recursos' => []
                    ]
                ], 200);
            }

            // Traer todos los archivos para los recursos obtenidos (una sola query)
            $recursoIds = $recursos->pluck('id_recurso')->toArray();

            $archivos = DB::table('recurso_archivos')
                ->whereIn('id_recurso', $recursoIds)
                ->select('id', 'id_recurso', 'ruta', 'nombre_original', 'tipo_archivo', 'orden')
                ->orderBy('orden', 'asc')
                ->get()
                ->groupBy('id_recurso'); // colecciones agrupadas por id_recurso

            // Mapear recursos para adjuntar sus archivos y normalizar rutas públicas
            $recursosConArchivos = $recursos->map(function ($r) use ($archivos) {
                $item = (array) $r;

                // normalizar portada a URL pública si es relativa
                $item['portada'] = $this->toAbsolutePublicUrl($item['portada'] ?? '');

                // adjuntos: si hay, convertir rutas relativas a URL públicas
                $files = [];
                if (!empty($archivos[$r->id_recurso])) {
                    foreach ($archivos[$r->id_recurso] as $f) {
                        $files[] = [
                            'id' => $f->id,
                            'ruta' => $this->toAbsolutePublicUrl($f->ruta),
                            'nombre_original' => $f->nombre_original,
                            'tipo_archivo' => $f->tipo_archivo,
                            'orden' => $f->orden,
                        ];
                    }
                }
                $item['archivos'] = $files;

                return (object) $item;
            });

            return response()->json([
                'message' => 'Recursos obtenidos correctamente.',
                'data' => [
                    'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                    'seccion' => ['id_seccion' => $seccion->id_seccion, 'nombre' => $seccion->nombre, 'nombre' => $seccion->nombre ?? null],
                    'categorias' => $categorias,
                    'recursos' => $recursosConArchivos
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error en RecursosPublicController@index', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener recursos públicos.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    public function bibliotecaRecursos(Request $request)
    {
        try {
            $projectKey = trim($request->query('project', 'sanamente'));
            $sectionKey = 'biblioteca de recursos'; // Forzamos la sección a Biblioteca de Recursos

            // Buscar proyecto por nombre (preferido) o por nombre (case-insensitive)
            $project = DB::table('proyectos')
                ->whereRaw('LOWER(nombre) = ?', [strtolower($projectKey)])
                ->orWhereRaw('LOWER(nombre) = ?', [strtolower($projectKey)])
                ->first();

            if (! $project) {
                // Si no existe el proyecto, devolvemos vacíos pero status 200 (público)
                return response()->json([
                    'message' => "Proyecto '{$projectKey}' no encontrado.",
                    'data' => [
                        'proyecto' => null,
                        'seccion' => null,
                        'categorias' => [],
                        'recursos' => []
                    ]
                ], 200);
            }

            // Buscar sección "Biblioteca de Recursos" dentro del proyecto
            $seccion = DB::table('secciones')
                ->where('id_proyecto', $project->id_proyecto)
                ->where(function ($q) use ($sectionKey) {
                    $q->whereRaw('LOWER(nombre) = ?', [strtolower($sectionKey)])
                      ->orWhereRaw('LOWER(nombre) = ?', [strtolower($sectionKey)]);
                })
                ->first();

            if (! $seccion) {
                return response()->json([
                    'message' => "Sección '{$sectionKey}' no encontrada en el proyecto '{$projectKey}'.",
                    'data' => [
                        'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                        'seccion' => null,
                        'categorias' => [],
                        'recursos' => []
                    ]
                ], 200);
            }

            // Obtener categorías de la sección (solo campos necesarios)
            $categorias = DB::table('categorias')
                ->where('id_seccion', $seccion->id_seccion)
                ->select('id_categoria', 'nombre', 'nombre')
                ->orderBy('nombre')
                ->get();

            // Determinar id del estado "aprobado" (buscar por nombre 'aprobado' case-insensitive)
            $approvedEstadoId = DB::table('estados')
                ->whereRaw('LOWER(nombre_estado) = ?', ['aprobado'])
                ->value('id_estado');

            // Si no encontramos el estado "aprobado", intentamos usar id = 2 como fallback (opcional)
            if (empty($approvedEstadoId)) {
                $approvedEstadoId = 2; // ajusta si en tu BD es otro id o maneja como error
            }

            // Obtener recursos aprobados del proyecto y sección, seleccionar solo campos necesarios
            $recursos = DB::table('recursos')
                ->where('id_proyecto', $project->id_proyecto)
                ->where('id_seccion', $seccion->id_seccion)
                ->where('id_estado', $approvedEstadoId)
                ->select(
                    'id_recurso',
                    'id_proyecto',
                    'id_seccion',
                    'id_categoria',
                    'titulo',
                    'descripcion',
                    'portada',
                    'url',
                    'created_at'
                )
                ->orderBy('created_at', 'desc')
                ->get();

            if ($recursos->isEmpty()) {
                return response()->json([
                    'message' => 'No hay recursos aprobados para esta sección/proyecto.',
                    'data' => [
                        'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                        'seccion' => ['id_seccion' => $seccion->id_seccion, 'nombre' => $seccion->nombre, 'nombre' => $seccion->nombre ?? null],
                        'categorias' => $categorias,
                        'recursos' => []
                    ]
                ], 200);
            }

            // Traer todos los archivos para los recursos obtenidos (una sola query)
            $recursoIds = $recursos->pluck('id_recurso')->toArray();

            $archivos = DB::table('recurso_archivos')
                ->whereIn('id_recurso', $recursoIds)
                ->select('id', 'id_recurso', 'ruta', 'nombre_original', 'tipo_archivo', 'orden')
                ->orderBy('orden', 'asc')
                ->get()
                ->groupBy('id_recurso'); // colecciones agrupadas por id_recurso

            // Mapear recursos para adjuntar sus archivos y normalizar rutas públicas
            $recursosConArchivos = $recursos->map(function ($r) use ($archivos) {
                $item = (array) $r;

                // normalizar portada a URL pública si es relativa
                $item['portada'] = $this->toAbsolutePublicUrl($item['portada'] ?? '');

                // adjuntos: si hay, convertir rutas relativas a URL públicas
                $files = [];
                if (!empty($archivos[$r->id_recurso])) {
                    foreach ($archivos[$r->id_recurso] as $f) {
                        $files[] = [
                            'id' => $f->id,
                            'ruta' => $this->toAbsolutePublicUrl($f->ruta),
                            'nombre_original' => $f->nombre_original,
                            'tipo_archivo' => $f->tipo_archivo,
                            'orden' => $f->orden,
                        ];
                    }
                }
                $item['archivos'] = $files;

                return (object) $item;
            });

            return response()->json([
                'message' => 'Recursos obtenidos correctamente.',
                'data' => [
                    'proyecto' => ['id_proyecto' => $project->id_proyecto, 'nombre' => $project->nombre, 'nombre' => $project->nombre ?? null],
                    'seccion' => ['id_seccion' => $seccion->id_seccion, 'nombre' => $seccion->nombre, 'nombre' => $seccion->nombre ?? null],
                    'categorias' => $categorias,
                    'recursos' => $recursosConArchivos
                ]
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error en RecursosPublicController@bibliotecaRecursos', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener recursos públicos de biblioteca.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Convierte rutas relativas/paths a URL pública tipo url('storage/...').
     * Si $candidate ya es URL absoluta, la devuelve tal cual.
     */
    private function toAbsolutePublicUrl(?string $candidate): ?string
    {
        if (empty($candidate)) return null;

        $candidate = trim($candidate);

        // Si ya es URL absoluta
        if (preg_match('/^https?:\\/\\//i', $candidate)) {
            return $candidate;
        }

        // Si viene con prefijo '/storage/' o 'storage/' -> extraer la parte relativa
        if (strpos($candidate, '/storage/') !== false) {
            $relative = ltrim(substr($candidate, strpos($candidate, '/storage/') + strlen('/storage/')), '/');
            return url('storage/' . $relative);
        }

        if (strpos($candidate, 'storage/') === 0) {
            $relative = ltrim(substr($candidate, strlen('storage/')), '/');
            return url('storage/' . $relative);
        }

        // Si parece un path relativo devuelto por $file->store(), ej: "recursos/101/archivo.jpg" o "portadas/123/.."
        return url('storage/' . ltrim($candidate, '/'));
    }
}