<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Illuminate\Http\UploadedFile;

class RevisionController extends Controller
{
    /**
     * Listar revisiones. Soporta filtros opcionales: id_recurso, id_user, id_estado.
     */
    public function index(Request $request)
    {
        try {
            $q = DB::table('revisiones')
                ->leftJoin('recursos', 'revisiones.id_recurso', '=', 'recursos.id_recurso')
                ->leftJoin('users', 'revisiones.id_user', '=', 'users.id_user')
                ->leftJoin('estados', 'revisiones.id_estado', '=', 'estados.id_estado')
                ->select(
                    'revisiones.id_revision',
                    'revisiones.id_recurso',
                    'recursos.titulo as recurso_titulo',
                    'revisiones.id_user',
                    'users.nombre as revisor_nombre',
                    'users.apellido_paterno as revisor_apellido_paterno',
                    'revisiones.id_estado',
                    'estados.nombre_estado as estado_nombre',
                    'revisiones.comentarios',
                    'revisiones.fecha_revision'
                );

            if ($request->filled('id_recurso')) {
                $q->where('revisiones.id_recurso', $request->id_recurso);
            }
            if ($request->filled('id_user')) {
                $q->where('revisiones.id_user', $request->id_user);
            }
            if ($request->filled('id_estado')) {
                $q->where('revisiones.id_estado', $request->id_estado);
            }

            $q->orderBy('revisiones.fecha_revision', 'desc');

            $revisiones = $q->get();

            return response()->json([
                'message' => 'Revisiones listadas correctamente.',
                'data' => $revisiones
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al listar revisiones', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener revisiones.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Mostrar una revisión por id_revision
     */
    public function show($id)
    {
        try {
            $rev = DB::table('revisiones')
                ->leftJoin('users', 'revisiones.id_user', '=', 'users.id_user')
                ->leftJoin('estados', 'revisiones.id_estado', '=', 'estados.id_estado')
                ->leftJoin('recursos', 'revisiones.id_recurso', '=', 'recursos.id_recurso')
                ->select(
                    'revisiones.id_revision',
                    'revisiones.id_recurso',
                    'recursos.titulo as recurso_titulo',
                    'revisiones.id_user',
                    'users.nombre as revisor_nombre',
                    'users.apellido_paterno as revisor_apellido_paterno',
                    'revisiones.id_estado',
                    'estados.nombre_estado as estado_nombre',
                    'revisiones.comentarios',
                    'revisiones.fecha_revision'
                )
                ->where('revisiones.id_revision', $id)
                ->first();

            if (! $rev) {
                return response()->json(['error' => "Revisión {$id} no existe."], 404);
            }

            return response()->json([
                'message' => 'Revisión obtenida.',
                'data' => $rev
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al obtener revisión', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener revisión.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Crear una revisión (aprobación/rechazo u otra transición).
     *
     * Request esperado:
     * - id_recurso (required, exists)
     * - id_estado (required, exists en tabla estados)
     * - comentarios (nullable)
     * - portada (nullable string o file) -> si viene y estado = 'aprobado' se asigna a recursos.portada
     *
     * Nota: este endpoint actualiza también la fila en 'recursos' (id_estado, approved_by/approved_at, portada).
     * Todo dentro de una transacción para garantizar atomicidad.
     */
    public function store(Request $request)
{
    Log::info('Inicio de store() en RevisionController', ['input' => $request->all()]);

    // validación base
    $validator = Validator::make($request->all(), [
        'id_recurso' => 'required|integer|exists:recursos,id_recurso',
        'id_estado' => 'required|integer|exists:estados,id_estado',
        'comentarios' => 'nullable|string',
        'portada' => 'nullable',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación base fallida al crear revisión', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors()
        ], 422);
    }

    $user = auth()->user();
    if (! $user) {
        return response()->json(['error' => 'No autorizado.'], 401);
    }
    $userId = $user->id_user;

    DB::beginTransaction();
    // para borrar portada antigua después del commit (si corresponde)
    $oldPortadaRelativeToDelete = null;

    try {
        $recurso = DB::table('recursos')->where('id_recurso', $request->id_recurso)->first();
        if (! $recurso) {
            DB::rollBack();
            return response()->json(['error' => 'Recurso no encontrado.'], 404);
        }

        $estadoNombre = DB::table('estados')->where('id_estado', $request->id_estado)->value('nombre_estado');

        // reglas condicionales
        if (strtolower(trim($estadoNombre)) === 'aprobado') {
            $hasFilePortada = $request->hasFile('portada') && $request->file('portada') instanceof UploadedFile;
            $hasStringPortada = $request->filled('portada') && ! $hasFilePortada;

            if (! $hasFilePortada && ! $hasStringPortada) {
                DB::rollBack();
                return response()->json([
                    'error' => 'Validación: cuando se aprueba, la portada es obligatoria (file o ruta).'
                ], 422);
            }

            if ($hasFilePortada) {
                $file = $request->file('portada');
                $fileValidator = Validator::make(['portada' => $file], [
                    'portada' => 'file|mimes:jpg,jpeg,png,webp,gif|max:5120'
                ]);
                if ($fileValidator->fails()) {
                    DB::rollBack();
                    return response()->json([
                        'error' => 'Portada inválida.',
                        'details' => $fileValidator->errors()
                    ], 422);
                }
            }
        }

        if (strtolower(trim($estadoNombre)) === 'rechazado') {
            if (! $request->filled('comentarios')) {
                DB::rollBack();
                return response()->json([
                    'error' => 'Validación: cuando se rechaza, los comentarios son obligatorios.'
                ], 422);
            }
        }

        // insertar la revisión
        $idRevision = DB::table('revisiones')->insertGetId([
            'id_recurso' => $request->id_recurso,
            'id_user' => $userId,
            'id_estado' => $request->id_estado,
            'comentarios' => $request->comentarios ?? null,
            'fecha_revision' => now(),
        ]);

        // preparar update del recurso
        $updateData = [
            'id_estado' => $request->id_estado,
            'updated_at' => now(),
        ];

        if (strtolower(trim($estadoNombre)) === 'aprobado') {
            $updateData['approved_by'] = $userId;
            $updateData['approved_at'] = now();

            // --- CAMBIO PRINCIPAL: Guardar la portada en "portadas/{idRecurso}/..."
            if ($request->hasFile('portada') && $request->file('portada') instanceof UploadedFile) {
                $file = $request->file('portada');

                // Opción: almacenar en storage/app/public/portadas/{idRecurso}/
                $path = $file->store("portadas/{$request->id_recurso}", 'public'); // ej: "portadas/101/portada.jpg"
                $rutaPublica = $this->toAbsolutePublicUrl($path);
                $updateData['portada'] = $rutaPublica;

                // programamos el borrado de la portada antigua (si existía y era manejable)
                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel && $oldRel !== $path) {
                        $oldPortadaRelativeToDelete = $oldRel;
                    }
                }
            } elseif ($request->filled('portada')) {
                // si envían ruta como string, convertir a URL pública (no movemos fichero)
                $updateData['portada'] = $this->toAbsolutePublicUrl($request->portada);

                // si la portada anterior estaba en storage y el cliente envía otra ruta pública,
                // decide si quieres eliminar la anterior: (opcional)
                if (!empty($recurso->portada)) {
                    $oldRel = $this->getRelativePathFromStoredRuta($recurso->portada);
                    if ($oldRel) $oldPortadaRelativeToDelete = $oldRel;
                }
            }
        }

        DB::table('recursos')->where('id_recurso', $request->id_recurso)->update($updateData);

        DB::commit();

        // --- BORRADO SEGURO DESPUÉS DEL COMMIT (opcional, recomendado)
        if (!empty($oldPortadaRelativeToDelete)) {
            try {
                Storage::disk('public')->delete($oldPortadaRelativeToDelete);
                Log::info("Portada antigua eliminada: {$oldPortadaRelativeToDelete}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar la portada antigua {$oldPortadaRelativeToDelete}: " . $ex->getMessage());
            }
        }

        $revision = DB::table('revisiones')->where('id_revision', $idRevision)->first();
        Log::info("Revisión creada ID: {$idRevision} para recurso {$request->id_recurso} por user {$userId}, estado {$request->id_estado}");

        return response()->json([
            'message' => 'Revisión registrada correctamente.',
            'data' => $revision
        ], 201);

    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Error al crear revisión, rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);

        // si hubo un archivo nuevo guardado y quieres eliminarlo en el caso de error,
        // necesitarías capturar su $path/rel y borrarlo aquí con Storage::disk('public')->delete($path);

        return response()->json([
            'error' => 'Error al registrar revisión.',
            'details' => $e->getMessage()
        ], 500);
    }
}

    /**
     * Convierte una ruta relativa/absoluta o un path relativo (devuelto por store()) a una URL pública absoluta.
     * Ej:
     * - "recursos/3/file.jpg" => url('storage/recursos/3/file.jpg')
     * - "/storage/recursos/3/file.jpg" => url('storage/recursos/3/file.jpg')
     * - "https://..." => se devuelve tal cual
     */
    private function toAbsolutePublicUrl(string $candidate = null)
    {
        if (empty($candidate)) return null;

        // Si ya es URL absoluta
        if (preg_match('/^https?:\\/\\//i', $candidate)) {
            return $candidate;
        }

        // Si viene con prefijo '/storage/' o 'storage/' -> extraer la parte relativa
        if (strpos($candidate, '/storage/') !== false) {
            $relative = ltrim(substr($candidate, strpos($candidate, '/storage/') + strlen('/storage/')), '/');
            return url('storage/' . $relative);
        }

        if (strpos($candidate, 'storage/') === 0) {
            $relative = ltrim(substr($candidate, strlen('storage/')), '/');
            return url('storage/' . $relative);
        }

        // Si parece un path relativo devuelto por $file->store()
        // ej: "recursos/101/archivo.jpg"
        return url('storage/' . ltrim($candidate, '/'));
    }
}
