<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class SeccionesController extends Controller
{
    /**
     * Listar secciones (opcional: incluir nombre del proyecto).
     */
    public function index()
    {
        Log::info('Inicio de index() en SeccionesController');

        try {
            $secciones = DB::table('secciones')
                ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
                ->select(
                    'secciones.id_seccion',
                    'secciones.id_proyecto',
                    'proyectos.nombre as proyecto_nombre',
                    'secciones.nombre',
                    'secciones.descripcion',
                    'secciones.orden',
                    'secciones.visible',
                    'secciones.created_at',
                    'secciones.updated_at'
                )
                ->orderBy('secciones.orden', 'asc')
                ->get();

           

            return response()->json([
                'message' => 'Secciones listadas correctamente.',
                'data' => $secciones
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al listar secciones', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener secciones.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Crear una sección
     */
    public function store(Request $request)
{
    Log::info('Inicio de store() en SeccionesController', ['input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'id_proyecto' => 'required|integer|exists:proyectos,id_proyecto',
        'nombre' => 'required|string|max:150',
        'descripcion' => 'nullable|string',
        'orden' => 'nullable|integer',
        'visible' => 'nullable|boolean',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al crear sección', ['errors' => $validator->errors()->all()]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors()
        ], 422);
    }

    // Verificar unicidad de nombre (por proyecto)
    Log::debug('Chequeo unicidad nombre antes de crear', ['id_proyecto' => $request->id_proyecto, 'nombre' => $request->nombre]);
    $exists = DB::table('secciones')
        ->where('id_proyecto', $request->id_proyecto)
        ->where('nombre', $request->nombre)
        ->exists();

    if ($exists) {
        Log::warning('Nombre duplicado al crear sección', ['id_proyecto' => $request->id_proyecto, 'nombre' => $request->nombre]);
        return response()->json(['error' => 'Ya existe una sección con ese nombre en el proyecto.'], 422);
    }

    DB::beginTransaction();
    Log::info('Transacción iniciada para crear sección');

    $createdDirs = [];

    try {
        $id = DB::table('secciones')->insertGetId([
            'id_proyecto' => $request->id_proyecto,
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion ?? null,
            'orden' => $request->orden ?? 0,
            'visible' => $request->filled('visible') ? (bool)$request->visible : true,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        // --- Crear carpeta para la sección dentro de la carpeta del proyecto ---
        $sectionDir = "proyectos/{$request->id_proyecto}/secciones/{$id}";
        \Illuminate\Support\Facades\Storage::disk('public')->makeDirectory($sectionDir);
        $createdDirs[] = $sectionDir;
        Log::info("Carpeta creada para sección: {$sectionDir}");

        DB::commit();
        Log::info('Transacción commit - sección creada', ['id_seccion' => $id]);

        // --- Obtener la sección creada con los mismos campos que en index() ---
        $seccion = DB::table('secciones')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->select(
                'secciones.id_seccion',
                'secciones.id_proyecto',
                'proyectos.nombre as proyecto_nombre',
                'secciones.nombre',
                'secciones.descripcion',
                'secciones.orden',
                'secciones.visible',
                'secciones.created_at',
                'secciones.updated_at'
            )
            ->where('secciones.id_seccion', $id)
            ->first();

        // Fallback: si por alguna razón la consulta con join no devuelve, recuperar la fila simple
        if (!$seccion) {
            Log::warning('Consulta con join no devolvió la sección; usando fallback simple.', ['id_seccion' => $id]);
            $seccion = DB::table('secciones')->where('id_seccion', $id)->first();
        }

        Log::info('Sección creada y recuperada para respuesta', ['id' => $id, 'seccion' => (array)$seccion]);

        return response()->json([
            'message' => 'Sección creada correctamente.',
            'data' => $seccion
        ], 201);
    } catch (\Exception $e) {
        DB::rollBack();

        foreach ($createdDirs as $dir) {
            try {
                \Illuminate\Support\Facades\Storage::disk('public')->deleteDirectory($dir);
                Log::info("Carpeta eliminada por rollback: {$dir}");
            } catch (\Exception $ex) {
                Log::warning("No se pudo eliminar la carpeta en rollback: {$dir}. Error: " . $ex->getMessage());
            }
        }

        Log::error('Error al crear sección - rollback', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
        return response()->json([
            'error' => 'Error al crear sección.',
            'details' => $e->getMessage()
        ], 500);
    }
}



    /**
     * Actualizar una sección
     */
    public function update(Request $request, $id)
{
    Log::info('Inicio de update() en SeccionesController', ['id' => $id, 'input' => $request->all()]);

    $validator = Validator::make($request->all(), [
        'id_proyecto' => 'nullable|integer|exists:proyectos,id_proyecto',
        'nombre' => 'required|string|max:150',
        'descripcion' => 'nullable|string',
        'orden' => 'nullable|integer',
        'visible' => 'nullable|boolean',
    ]);

    if ($validator->fails()) {
        Log::warning('Validación fallida al actualizar sección', ['errors' => $validator->errors()->all(), 'id' => $id]);
        return response()->json([
            'error' => 'Datos inválidos.',
            'details' => $validator->errors()
        ], 422);
    }

    DB::beginTransaction();
    Log::info('Transacción iniciada para actualizar sección', ['id' => $id]);

    try {
        $row = DB::table('secciones')->where('id_seccion', $id)->first();
        if (! $row) {
            Log::warning('Intento de actualizar sección inexistente', ['id' => $id]);
            DB::rollBack();
            return response()->json(['error' => "Sección {$id} no existe."], 404);
        }

        $idProyecto = $request->filled('id_proyecto') ? $request->id_proyecto : $row->id_proyecto;

        // Verificar unicidad de nombre dentro del proyecto (excluir la propia fila)
        Log::debug('Chequeo unicidad nombre antes de actualizar', ['id_proyecto' => $idProyecto, 'nombre' => $request->nombre, 'exclude_id' => $id]);
        $exists = DB::table('secciones')
            ->where('id_proyecto', $idProyecto)
            ->where('nombre', $request->nombre)
            ->where('id_seccion', '!=', $id)
            ->exists();

        if ($exists) {
            Log::warning('Nombre duplicado al actualizar sección', ['id_proyecto' => $idProyecto, 'nombre' => $request->nombre, 'exclude_id' => $id]);
            DB::rollBack();
            return response()->json(['error' => 'Ya existe una sección con ese nombre en el proyecto.'], 422);
        }

        DB::table('secciones')->where('id_seccion', $id)->update([
            'id_proyecto' => $idProyecto,
            'nombre' => $request->nombre,
            'descripcion' => $request->descripcion ?? $row->descripcion,
            'orden' => $request->filled('orden') ? $request->orden : $row->orden,
            'visible' => $request->filled('visible') ? (bool)$request->visible : $row->visible,
            'updated_at' => now(),
        ]);

        DB::commit();
        Log::info("Transacción commit - sección actualizada", ['id_seccion' => $id]);

        // --- Obtener la sección actualizada con los mismos campos que en index() ---
        $seccion = DB::table('secciones')
            ->leftJoin('proyectos', 'secciones.id_proyecto', '=', 'proyectos.id_proyecto')
            ->select(
                'secciones.id_seccion',
                'secciones.id_proyecto',
                'proyectos.nombre as proyecto_nombre',
                'secciones.nombre',
                'secciones.descripcion',
                'secciones.orden',
                'secciones.visible',
                'secciones.created_at',
                'secciones.updated_at'
            )
            ->where('secciones.id_seccion', $id)
            ->first();

        // Fallback: si por alguna razón la consulta con join no devuelve, recuperar la fila simple
        if (! $seccion) {
            Log::warning('Consulta con join no devolvió la sección actualizada; usando fallback simple.', ['id_seccion' => $id]);
            $seccion = DB::table('secciones')->where('id_seccion', $id)->first();
        }

        Log::info("Sección ID {$id} actualizada (respuesta)", ['before' => $row, 'after' => (array)$seccion]);

        return response()->json([
            'message' => 'Sección actualizada correctamente.',
            'data' => $seccion
        ], 200);
    } catch (\Exception $e) {
        DB::rollBack();
        Log::error('Error al actualizar sección', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString(), 'id' => $id]);
        return response()->json([
            'error' => 'Error al actualizar sección.',
            'details' => $e->getMessage()
        ], 500);
    }
}

}
