<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Log;

class TiposPublicacionController extends Controller
{
    /**
     * Listar tipos de publicación
     */
    public function index()
    {
        try {
            $tipos = DB::table('tipo_publicaciones')
                ->select('id_tipo_publicacion', 'nombre', 'descripcion', 'created_at', 'updated_at')
                ->orderByDesc('id_tipo_publicacion')
                ->get();

            return response()->json([
                'message' => 'Tipos de publicación listados correctamente.',
                'data' => $tipos
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error al listar tipos de publicación', ['error' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al obtener tipos de publicación.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Crear un tipo de publicación
     */
    public function store(Request $request)
    {
        Log::info('Inicio de store() en TiposPublicacionController', ['input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'nombre' => 'required|string|max:100|unique:tipo_publicaciones,nombre',
            'descripcion' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al crear tipo de publicación', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $id = DB::table('tipo_publicaciones')->insertGetId([
                'nombre' => $request->nombre,
                'descripcion' => $request->descripcion ?? null,
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::commit();

            $tipo = DB::table('tipo_publicaciones')->where('id_tipo_publicacion', $id)->first();

            Log::info("Tipo de publicación creado ID: {$id}");

            return response()->json([
                'message' => 'Tipo de publicación creado correctamente.',
                'data' => $tipo
            ], 201);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al crear tipo de publicación', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al crear tipo de publicación.',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar un tipo de publicación
     */
    public function update(Request $request, $id)
    {
        Log::info('Inicio de update() en TiposPublicacionController', ['id' => $id, 'input' => $request->all()]);

        $validator = Validator::make($request->all(), [
            'nombre' => "required|string|max:100|unique:tipo_publicaciones,nombre,{$id},id_tipo_publicacion",
            'descripcion' => 'nullable|string|max:1000',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al actualizar tipo de publicación', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $row = DB::table('tipo_publicaciones')->where('id_tipo_publicacion', $id)->first();
            if (! $row) {
                return response()->json(['error' => "Tipo de publicación {$id} no existe."], 404);
            }

            DB::table('tipo_publicaciones')->where('id_tipo_publicacion', $id)->update([
                'nombre' => $request->nombre,
                'descripcion' => $request->descripcion ?? $row->descripcion,
                'updated_at' => now(),
            ]);

            DB::commit();

            $tipo = DB::table('tipo_publicaciones')->where('id_tipo_publicacion', $id)->first();
            Log::info("Tipo de publicación ID {$id} actualizado");

            return response()->json([
                'message' => 'Tipo de publicación actualizado correctamente.',
                'data' => $tipo
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error al actualizar tipo de publicación', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al actualizar tipo de publicación.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
}
