<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Rol;
use App\Models\EstadoUser;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use Carbon\Carbon;
use Illuminate\Support\Facades\Validator;

class UserController extends Controller
{
    //--------Inicio de sesión-----------
    public function login(Request $request)
{
    Log::info('Intento de login recibido', [
        'nombre_usuario' => $request->nombre_usuario,
        'ip' => $request->ip(),
        'time' => now()
    ]);

    $request->validate([
        'nombre_usuario' => 'required|string',
        'password' => 'required|string',
    ]);

    $user = User::where('nombre_usuario', $request->nombre_usuario)->first();

    if (!$user) {
        Log::warning('Intento de login fallido - usuario no encontrado', [
            'nombre_usuario' => $request->nombre_usuario,
            'ip' => $request->ip()
        ]);
        return response()->json(['message' => 'Credenciales inválidas'], 401);
    }

    if (!Hash::check($request->password, $user->password)) {
        Log::warning('Intento de login fallido - contraseña incorrecta', [
            'user_id' => $user->id_user,
            'nombre_usuario' => $user->nombre_usuario,
            'ip' => $request->ip()
        ]);
        return response()->json(['message' => 'Credenciales inválidas'], 401);
    }

    if ($user->id_estado_user != 1) {
        Log::notice('Usuario inactivo/suspendido intentó iniciar sesión', [
            'user_id' => $user->id_user,
            'nombre_usuario' => $user->nombre_usuario,
            'estado' => $user->id_estado_user,
            'ip' => $request->ip()
        ]);
        return response()->json(['message' => 'Usuario inactivo o suspendido'], 403);
    }

    $tokenResult = $user->createToken('auth_token');
    $accessToken = $tokenResult->accessToken;
    $expiresAt = optional($tokenResult->token)->expires_at;

    Log::info('Inicio de sesión exitoso', [
        'user_id' => $user->id_user,
        'nombre_usuario' => $user->nombre_usuario,
        'ip' => $request->ip(),
        'expires_at' => $expiresAt
    ]);

    return response()->json([
        'message' => 'Inicio de sesión exitoso',
        'token' => $accessToken,
        'expires_at' => $expiresAt,
        'nombre_usuario' => $user->nombre,
        'telefono' => $user->telefono,
        'email' => $user->email,
        'rol' => $user->rol->nombre_rol,
        'id_user' => $user->id_user
    ], 200);
}

    //------Cierre de sesión------
    public function logout(Request $request)
    {
        // Si existe el token actual, lo revocamos. Si no, eliminamos todos los tokens del usuario.
        try {
            $user = $request->user();
            if ($user) {
                if (method_exists($request->user(), 'token') && $request->user()->token()) {
                    // revocar token actual
                    $request->user()->token()->revoke();
                } else {
                    // eliminar todos (fallback)
                    $request->user()->tokens()->delete();
                }
            }

            return response()->json(['message' => 'Cierre de sesión exitoso'], 200);
        } catch (\Exception $e) {
            return response()->json(['message' => 'Error al cerrar sesión: ' . $e->getMessage()], 500);
        }
    }

    //-----Obtener información del usuario autenticado------
    public function me(Request $request)
{
    $user = $request->user()->load('rol'); // carga la relación rol

    return response()->json([
        'id_user' => $user->id_user,
        'nombre_usuario' => $user->nombre_usuario,
        'nombre' => $user->nombre,
        'apellido_paterno' => $user->apellido_paterno,
        'apellido_materno' => $user->apellido_materno,
        'telefono' => $user->telefono,
        'ci' => $user->ci,
        'email' => $user->email,
        'id_rol' => $user->id_rol,
        'rol' => $user->rol ? $user->rol->nombre_rol : null, // <- importante
    ], 200);
}



    //-----Crear un nuevo usuario (SuperAdmin)--------
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'nombre_usuario' => 'required|unique:users',
            'password' => 'required',
            'nombre' => 'required',
            'apellido_paterno' => 'required',
            'apellido_materno' => 'required',
            'email' => 'required|email|unique:users',
            'telefono' => 'required',
            'ci' => 'required',
            'id_rol' => 'required|exists:roles,id_rol',
            'id_entidad' => 'required|exists:entidades,id_entidad',
        ]);

        $user = User::create([
            'nombre_usuario' => $validatedData['nombre_usuario'],
            'password' => Hash::make($validatedData['password']),
            'nombre' => $validatedData['nombre'],
            'apellido_paterno' => $validatedData['apellido_paterno'],
            'apellido_materno' => $validatedData['apellido_materno'],
            'email' => $validatedData['email'],
            'telefono' => $validatedData['telefono'],
            'ci' => $validatedData['ci'],
            'id_rol' => $validatedData['id_rol'],
            'id_entidad' => $validatedData['id_entidad'],
            'id_estado_user' => 1, 
        ]);

        return response()->json(['message' => 'Usuario creado exitosamente', 'user' => $user], 201);
    }



    //-----Actualizar un usuario------
    public function update(Request $request, $id)
{
    $validated = $request->validate([
        'nombre_usuario' => 'required|string|max:255',
        'email' => 'required|email|max:255|unique:users,email,' . $id . ',id_user',
        'telefono' => 'nullable|string|max:20',
        'password' => 'nullable|string|min:8',
        'nombre' => 'nullable|string|max:255',
        'apellido_paterno' => 'nullable|string|max:255',
        'apellido_materno' => 'nullable|string|max:255',
        'ci' => 'nullable|string|max:20',
        'id_entidad' => 'nullable|exists:entidades,id',
        'id_rol' => 'nullable|exists:roles,id',
        'id_estado_user' => 'nullable|exists:estado_users,id_estado_user', // Validación para el estado
    ]);

    try {
        $user = User::findOrFail($id);
        $user->nombre_usuario = $validated['nombre_usuario'];
        $user->email = $validated['email'];

        $user->telefono = $validated['telefono'] ?? $user->telefono;
        $user->nombre = $validated['nombre'] ?? $user->nombre;
        $user->apellido_paterno = $validated['apellido_paterno'] ?? $user->apellido_paterno;
        $user->apellido_materno = $validated['apellido_materno'] ?? $user->apellido_materno;
        $user->ci = $validated['ci'] ?? $user->ci;
        $user->id_entidad = $validated['id_entidad'] ?? $user->id_entidad;
        $user->id_rol = $validated['id_rol'] ?? $user->id_rol;

        //-----Solo actualizar el password si fue enviado-------
        if ($request->filled('password')) {
            $user->password = bcrypt($validated['password']);
        }

        //-----Actualizar el estado del usuario si fue enviado-------
        if (isset($validated['id_estado_user'])) {
            $user->id_estado_user = $validated['id_estado_user'];
        }

        $user->save();

        return response()->json(['user' => $user], 200);
    } catch (\Exception $e) {
        return response()->json(['error' => 'Error al actualizar el usuario: ' . $e->getMessage()], 500);
    }
}

public function update2(Request $request, $id)
    {
        Log::info('Inicio de update() UsersController', ['id' => $id, 'input' => $request->all()]);

        // Reglas dinámicas para unique (ignorar el propio id_user)
        $validator = Validator::make($request->all(), [
            'nombre_usuario'   => "required|string|max:100|unique:users,nombre_usuario,{$id},id_user",
            'password'         => 'nullable|string|min:6',
            'nombre'           => 'required|string|max:100',
            'apellido_paterno' => 'required|string|max:100',
            'apellido_materno' => 'nullable|string|max:100',
            'telefono'         => 'nullable|string|max:20',
            'ci'               => 'nullable|string|max:20',
            'email'            => "required|email|max:150|unique:users,email,{$id},id_user",
            'id_estado_user'   => 'required|exists:estado_users,id_estado_user',
            'id_entidad'       => 'nullable|exists:entidades,id_entidad',
            'id_rol'           => 'required|exists:roles,id_rol',
        ]);

        if ($validator->fails()) {
            Log::warning('Validación fallida al actualizar usuario', ['errors' => $validator->errors()->all()]);
            return response()->json([
                'error' => 'Datos inválidos.',
                'details' => $validator->errors()
            ], 422);
        }

        DB::beginTransaction();
        try {
            $user = DB::table('users')->where('id_user', $id)->first();
            if (!$user) {
                Log::error("Usuario no encontrado ID: {$id}");
                return response()->json(['error' => "Usuario {$id} no existe."], 404);
            }

            $dataToUpdate = [
                'nombre_usuario'     => $request->nombre_usuario,
                'nombre'             => $request->nombre,
                'apellido_paterno'   => $request->apellido_paterno,
                'apellido_materno'   => $request->apellido_materno ?? null,
                'telefono'           => $request->telefono ?? null,
                'ci'                 => $request->ci ?? null,
                'email'              => $request->email,
                'id_estado_user'     => $request->id_estado_user,
                'id_entidad'         => $request->id_entidad ?? null,
                'id_rol'             => $request->id_rol,
                'updated_at'         => Carbon::now(),
            ];

            // Si enviaron password, actualizar (hash)
            if ($request->filled('password')) {
                $dataToUpdate['password'] = Hash::make($request->password);
            }

            // Si enviaron ultima_sesion explícita (opcional)
            if ($request->has('ultima_sesion')) {
                $dataToUpdate['ultima_sesion'] = $request->ultima_sesion;
            }

            DB::table('users')->where('id_user', $id)->update($dataToUpdate);

            DB::commit();

            $updatedUser = DB::table('users')
                ->leftJoin('roles', 'users.id_rol', '=', 'roles.id_rol')
                ->leftJoin('estado_users', 'users.id_estado_user', '=', 'estado_users.id_estado_user')
                ->leftJoin('entidades', 'users.id_entidad', '=', 'entidades.id_entidad')
                ->select(
                    'users.id_user',
                    'users.nombre_usuario',
                    'users.nombre',
                    'users.apellido_paterno',
                    'users.apellido_materno',
                    'users.telefono',
                    'users.ci',
                    'users.email',
                    'users.ultima_sesion',
                    'users.id_estado_user',
                    'estado_users.nombre_estado as estado',
                    'users.id_entidad',
                    'entidades.nombre_entidad as entidad',
                    'users.id_rol',
                    'roles.nombre_rol as rol',
                    'users.created_at',
                    'users.updated_at'
                )
                ->where('users.id_user', $id)
                ->first();

            return response()->json([
                'message' => 'Usuario actualizado correctamente.',
                'data' => $updatedUser
            ], 200);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error actualizando usuario', ['exception' => $e->getMessage(), 'trace' => $e->getTraceAsString()]);
            return response()->json([
                'error' => 'Error al actualizar usuario.',
                'details' => $e->getMessage()
            ], 500);
        }
    }




    //-----Eliminar un usuario------
    public function destroy($id)
    {
        $user = User::findOrFail($id);
    
        if ($user->rol && $user->rol->nombre_rol === 'SuperAdmin') {
            return response()->json(['message' => 'No puedes eliminar un SuperAdmin'], 403);
        }
    
        $user->delete();
        return response()->json(['message' => 'Usuario eliminado correctamente']);
    }
    

     //-----Listar todos los usuarios (Accesible solo para SuperAdmin)------
     public function index()
    {
        try {
            $users = DB::table('users')
                ->leftJoin('roles', 'users.id_rol', '=', 'roles.id_rol')
                ->leftJoin('estado_users', 'users.id_estado_user', '=', 'estado_users.id_estado_user')
                ->leftJoin('entidades', 'users.id_entidad', '=', 'entidades.id_entidad')
                ->select(
                    'users.id_user',
                    'users.nombre_usuario',
                    'users.nombre',
                    'users.apellido_paterno',
                    'users.apellido_materno',
                    'users.telefono',
                    'users.ci',
                    'users.email',
                    'users.ultima_sesion',
                    'users.id_estado_user',
                    'estado_users.nombre_estado',
                    'users.id_entidad',
                    'entidades.nombre_entidad',
                    'users.id_rol',
                    'roles.nombre_rol',
                    'users.created_at',
                    'users.updated_at'
                )
                ->orderBy('users.id_user', 'desc')
                ->get();

            return response()->json([
                'message' => 'Usuarios listados correctamente.',
                'users' => $users
            ], 200);
        } catch (\Exception $e) {
            Log::error('Error listando usuarios', ['error' => $e->getMessage()]);
            return response()->json([
                'error' => 'Error al listar usuarios.',
                'details' => $e->getMessage()
            ], 500);
        }
    }
     


  //-----Buscar por id----
  public function show($id)
    {
        try {
            $user = User::with(['estado:id_estado_user,nombre_estado', 'rol:id_rol,nombre_rol'])
                        ->findOrFail($id); // Esto lanzará una excepción si el usuario no existe

            return response()->json(['user' => $user], 200);
        } catch (\Exception $e) {
            return response()->json(['error' => 'Usuario no encontrado o error en la consulta: ' . $e->getMessage()], 500);
        }
    }


    
    public function changePassword(Request $request)
    {
    
    $request->validate([
        'current_password' => 'required|string',
        'new_password' => 'required|string|min:8|confirmed', 
    ]);

  
    $user = Auth::user();

   
    if (!Hash::check($request->current_password, $user->password)) {
        return response()->json(['message' => 'La contraseña actual no es correcta'], 403);
    }

    
    $user->password = Hash::make($request->new_password);
    $user->save();

    return response()->json(['message' => 'Contraseña actualizada exitosamente'], 200);
    }


}
