<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    public function up(): void
    {
        Schema::create('recursos', function (Blueprint $table) {
            $table->bigIncrements('id_recurso');

            // Relación principal con proyecto (recomendado para consultas rápidas y coherencia)
            $table->unsignedBigInteger('id_proyecto');

            // Sección y categoría (nullable porque puede que el recurso no tenga categoría)
            $table->unsignedBigInteger('id_seccion')->nullable();
            $table->unsignedBigInteger('id_categoria')->nullable();

            // Opcionales / metadatos

            $table->string('titulo', 255);
            $table->string('descripcion', 1000)->nullable();
            $table->string('url', 1000)->nullable();
            $table->string('portada', 1000)->nullable();

            // Estado y autores
            $table->unsignedBigInteger('id_estado')->default(1);
            // created_by debe ser nullable para evitar borrado en cascada de recursos al borrar usuario
            $table->unsignedBigInteger('created_by')->nullable();
            $table->unsignedBigInteger('approved_by')->nullable();
            $table->timestamp('approved_at')->nullable();

            // Referencia a la fila de integrantes (opcional — garantiza integridad si la usás)
            $table->unsignedBigInteger('id_integrante')->nullable();

            $table->timestamps();
            $table->softDeletes();

            /*
             * Índices
             */
            $table->index('id_proyecto', 'idx_recursos_proyecto');
            $table->index('id_seccion', 'idx_recursos_seccion');
            $table->index('id_categoria', 'idx_recursos_categoria');
            $table->index('id_estado', 'idx_recursos_estado');
            $table->index('created_by', 'idx_recursos_created_by');
            $table->index('id_integrante', 'idx_recursos_integrante');

            // Unicidad: recomiendo unicidad por proyecto + título (evita colisiones globales)
            $table->unique(['id_proyecto', 'titulo'], 'uq_recursos_proyecto_titulo');

            /*
             * Foreign keys
             *
             * - id_proyecto: ON DELETE CASCADE (si borrás proyecto, borrás su contenido).
             * - id_seccion / id_categoria: ON DELETE SET NULL (mantener recurso aunque se elimine la categoría/ sección).
             * - tipo_publicacion_id: RESTRICT (o SET NULL según tu política).
             * - created_by / approved_by: SET NULL para evitar borrar recursos cuando se borra usuario.
             * - id_integrante: SET NULL (opcional; si poblás este campo garantiza que quien creó era integrante).
             */
            $table->foreign('id_proyecto', 'fk_recursos_proyecto')
                ->references('id_proyecto')->on('proyectos')->onDelete('cascade');

            $table->foreign('id_seccion', 'fk_recursos_seccion')
                ->references('id_seccion')->on('secciones')->onDelete('set null');

            $table->foreign('id_categoria', 'fk_recursos_categoria')
                ->references('id_categoria')->on('categorias')->onDelete('set null');



            $table->foreign('id_estado', 'fk_recursos_estado')
                ->references('id_estado')->on('estados')->onDelete('restrict');

            $table->foreign('created_by', 'fk_recursos_created_by')
                ->references('id_user')->on('users')->onDelete('set null');

            $table->foreign('approved_by', 'fk_recursos_approved_by')
                ->references('id_user')->on('users')->onDelete('set null');

            // FK opcional a la tabla 'integrantes' (su PK en tu migración es 'id')
            $table->foreign('id_integrante', 'fk_recursos_integrante')
                ->references('id')->on('integrantes')->onDelete('set null');
        });
    }

    public function down(): void
    {
        Schema::table('recursos', function (Blueprint $table) {
            try { $table->dropForeign('fk_recursos_proyecto'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_seccion'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_categoria'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_estado'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_created_by'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_approved_by'); } catch (\Exception $e) {}
            try { $table->dropForeign('fk_recursos_integrante'); } catch (\Exception $e) {}
        });

        Schema::dropIfExists('recursos');
    }
};
